<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       http://www.visual4.de/
 * @since      1.0.0
 *
 * @package    V4_Wp_Webhook
 * @subpackage V4_Wp_Webhook/admin
 */

require_once __DIR__ . '/class-v4-wp-webhook-customer.php';
require_once __DIR__ . '/class-v4-wp-webhook-alt-address.php';
require_once ABSPATH . 'wp-admin/includes/user.php';

use v4\wp\plugins\wp1crm\Customer as v4Customer;

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    V4_Wp_Webhook
 * @subpackage V4_Wp_Webhook/admin
 * @author     visual4 <info@visual4.de>
 */
class V4_Wp_Webhook_Admin
{

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string $plugin_name The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string $version The current version of this plugin.
     */
    private $version;

    /**
     * @var array $register_fields
     */
    private $register_fields = [];

    /** @var array $countries */
    private $countries = [];

    /** @var array $states */
    private $states = [];

    private $wpwh_options = [];

    /**
     * Initialize the class and set its properties.
     *
     * @param string $plugin_name The name of this plugin.
     * @param string $version The version of this plugin.
     * @since    1.0.0
     */
    public function __construct($plugin_name, $version)
    {
        $this->plugin_name = $plugin_name;
        $this->version = $version;

        add_action('init', [$this, 'init']);

		$active_plugins = apply_filters('active_plugins', get_option('active_plugins'));
		$is_v4oauth_active = in_array('v4-oauth/v4oAuth.php', $active_plugins);
		$is_em_woo_active = in_array('events-manager-woocommerce/events-manager-woocommerce.php', $active_plugins);

		//em
		add_filter('pre_option_dbem_booking_feedback_email_exists', [$this, 'pre_option_dbem_booking_feedback_email_exists'], 10, 1);
		add_filter('em_register_new_user_pre', [$this, 'new_customer_data']);
		add_filter('em_registration_errors', [$this, 'registration_check_email'], 50, 3);
		add_filter('emp_form_validate_field', [$this, 'emp_validate_use_wpwh_register_fields'], 50, 4);
		if($is_em_woo_active) add_filter('option_dbem_multiple_bookings', function () {return 0;}, 999, 0);
		if($is_v4oauth_active) add_filter('option_dbem_bookings_registration_disable', function () {return 0;}, 999, 0);
		if($is_v4oauth_active) add_filter('option_dbem_bookings_registration_disable_user_emails', function () {return 0;}, 999, 0);
		add_action('personal_options_update', array($this, 'em_customer_send'), 999);
		add_action('edit_user_profile_update', array($this, 'em_customer_send'), 999);

		//woo
		add_filter('woocommerce_registration_errors', [$this, 'registration_check_email'], 50, 3);
		add_filter('woocommerce_new_customer_data', [$this, 'new_customer_data'], 50, 1);
		add_action('woocommerce_new_customer', [$this, 'wc_customer_send'], 50, 2);
		add_action('woocommerce_update_customer', [$this, 'wc_customer_send'], 50, 2);
		add_filter('option_woocommerce_registration_generate_username', function () {return 'yes';}, 999, 0);
		add_filter('option_woocommerce_registration_generate_password', function () {return 'yes';}, 999, 0);

		//wp1crm
		add_action('v4/wp/plugins/wp1crm/display_tools', [$this, 'display_tools']);
		add_action('wp_ajax_wp1crm_cleanup_serialized_meta', [$this, 'cleanup_serialized_meta']);

		new \v4\wp\plugins\wp1crm\AltAddress();

        global $V4_Wp_Webhook_Admin;
        $V4_Wp_Webhook_Admin = $this;

        // wp rest api endpoints
        include_once 'rest_api/init.php';

    }

    public function init()
    {
		$this->wpwh_options = get_option('v4_wpwh');
        $this->set_countries();
        $this->set_states();
        $this->set_register_fields();
    }

	public function pre_option_dbem_booking_feedback_email_exists()
	{
		return sprintf(__("The given e-mail address is already registered with us. If you have forgotten your password, please use the <a href='%s'>Forgot password?</a> function on the login page", "v4-wp-webhook"), '?lost_password=1crm');
	}

	public function new_customer_data($data)
	{
		$username = wp_generate_uuid4();
		$password = wp_generate_password();
		$data['user_login'] = $username;
		$data['user_pass']  = $password;
		return $data;
	}

	/**
	 * @param WP_Error $errors
	 * @param $username
	 * @param $email
	 * @return WP_Error
	 */
	public function registration_check_email(WP_Error $errors, $username, $email): WP_Error
	{
		if(email_exists($email)){
			$url = in_array('v4-oauth/v4oAuth.php', apply_filters('active_plugins', get_option('active_plugins'))) ? '?lost_password=1crm' : '/wp-admin';
			$msg = sprintf(__("The given e-mail address is already registered with us. If you have forgotten your password, please use the <a href='%s'>Forgot password?</a> function on the login page", "v4-wp-webhook"), $url);
			$errors->add( 'registration-error-email-exists', $msg );
		}

		return $errors;
	}

	/**
	 * @param $result
	 * @param $field
	 * @param $value
	 * @param EM_Form $EM_Forms
	 * @return bool|mixed
	 */
	public function emp_validate_use_wpwh_register_fields($result, $field, $value, EM_Form $EM_Form)
	{
		if(isset($this->wpwh_options['use_wpwh_register_fields']) && $this->wpwh_options['use_wpwh_register_fields']){
			$err = sprintf($EM_Form->form_required_error, $field['label']);
			$err_pos = array_search($err, $EM_Form->errors);
			if($err_pos !== false)
				unset($EM_Form->errors[$err_pos]);
			return true;
		}
		return $result;
	}

	/**
	 * @throws Exception
	 */
	public function wc_customer_send($customer_id, $WC_Customer)
	{
		$v4Customer = new v4Customer($WC_Customer);
		$v4Customer->save();
	}

	/**
	 * @throws Exception
	 */
	public function em_customer_send($user_id)
	{
		if(class_exists('EM_Person')) {
			try {
				$EM_Person = new EM_Person($user_id);
				$v4Customer = new v4Customer($EM_Person);
				$v4Customer->save();
			} catch (Exception $e) {
				if (is_admin()) {
					add_action('admin_notices', [$this, 'admin_notice__error_user']);
					error_log('v4-wp-webhook error (' . $e->getCode() . '): ' .  $e->getMessage());
				} else {
//					wp_delete_user($user_id);
				}
				return new WP_Error($e->getCode(), $e->getMessage());
			}
		}
		return $user_id;
	}

	public function admin_notice__error_user(): void
	{
		?>
		<div class="notice notice-error is-dismissible">
			<p>Es ist ein Fehler im Benutzer aufgetreten. Bitte prüfe das Error-Log.</p>
		</div>
		<?php
	}

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles()
    {

        /**
         * This function is provided for demonstration purposes only.
         *
         * An instance of this class should be passed to the run() function
         * defined in V4_Wp_Webhook_Loader as all of the hooks are defined
         * in that particular class.
         *
         * The V4_Wp_Webhook_Loader will then create the relationship
         * between the defined hooks and the functions defined in this
         * class.
         */
		wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/v4-wp-webhook-admin.css', array(), $this->version, 'all');
		wp_enqueue_style($this->plugin_name . '-public', plugin_dir_url(__FILE__) . '../public/css/v4-wp-webhook-public-loading.css', array(), $this->version, 'all');

		$active_plugins = apply_filters('active_plugins', get_option('active_plugins'));

		$is_v4oauth_active = in_array('v4-oauth/v4oAuth.php', $active_plugins);
		if($is_v4oauth_active)
			wp_enqueue_style($this->plugin_name . '-v4oauth', plugin_dir_url(__FILE__) . 'css/v4-wp-webhook-admin-v4oauth.css', array(), $this->version, 'all');

		$is_em_woo_active = in_array('events-manager-woocommerce/events-manager-woocommerce.php', $active_plugins);
		if($is_em_woo_active)
			wp_enqueue_style($this->plugin_name . '-em_woo', plugin_dir_url(__FILE__) . 'css/v4-wp-webhook-admin-em_woo.css', array(), $this->version, 'all');

    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts()
    {

        /**
         * This function is provided for demonstration purposes only.
         *
         * An instance of this class should be passed to the run() function
         * defined in V4_Wp_Webhook_Loader as all of the hooks are defined
         * in that particular class.
         *
         * The V4_Wp_Webhook_Loader will then create the relationship
         * between the defined hooks and the functions defined in this
         * class.
         */

        wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/v4-wp-webhook-admin.js', array('jquery'), $this->version, false);

        //array
        $parameter = [];
        //event
        if(is_admin() && isset($_REQUEST['action']) && $_REQUEST['action'] == 'edit' && isset($_REQUEST['post']) && get_post_type($_REQUEST['post']) == 'event' && class_exists('EM_Event')){
            global $post;
            //ticket irrelevant spaces
            $EM_Event = new EM_Event($post);
            foreach ($EM_Event->get_tickets()->tickets as $EM_Ticket){
                /** EM_Ticket $EM_Ticket */
                if(isset($EM_Ticket->ticket_meta['event_spaces_irrelevant']))
                    $event_spaces_irrelevant = $EM_Ticket->ticket_meta['event_spaces_irrelevant'];
                else
                    $event_spaces_irrelevant = get_post_meta($EM_Ticket->get_event()->post_id, 'ticket' . $EM_Ticket->ticket_id . '_event_spaces_irrelevant', true);
                $parameter['tickets'][] = [
                    'id' => $EM_Ticket->ticket_id,
                    'event_spaces_irrelevant' => $event_spaces_irrelevant
                ];
            }
        }
        if(!empty($parameter))
            wp_localize_script($this->plugin_name, 'v4em', $parameter);
    }

    /**
     * @param EM_Ticket $EM_Ticket
     */
    public function add_custom_data_to_ticket($EM_Ticket)
    {
		if(isset($_REQUEST['em_tickets']))
       		foreach ($_REQUEST['em_tickets'] as $ticket_req) {
				// match current ticket with tickets in request. if matched, edit event_spaces_irrelevant.
				if (
					(
						!empty($ticket_req['ticket_id']) &&
						!empty($EM_Ticket->ticket_id) &&
						$ticket_req['ticket_id'] == $EM_Ticket->ticket_id
					) || ( // either both have ids, or both don't have ids. Then we can match
						empty($ticket_req['ticket_id']) &&
						empty($EM_Ticket->ticket_id) &&
						$ticket_req['ticket_name'] == $EM_Ticket->ticket_name &&
						//$ticket_req['ticket_price'] == $EM_Ticket->ticket_price && // 0,00!=0.00 therefore dont check prices
						intval($ticket_req['ticket_spaces']) == $EM_Ticket->ticket_spaces
					)
				) {
					if (isset($ticket_req['ticket_event_spaces_irrelevant']) && $ticket_req['ticket_event_spaces_irrelevant'] == 'on')
						$EM_Ticket->ticket_meta['event_spaces_irrelevant'] = true;
					else
						$EM_Ticket->ticket_meta['event_spaces_irrelevant'] = false;
				}
			}
    }

    /**
     * @param $available_spaces
     * @param EM_Bookings $EM_Bookings
     * @return int
     */
    public function get_booking_relevant_spaces($available_spaces, $EM_Bookings)
    {
        /**
         * @var EM_Booking $EM_Booking
         * @var EM_Ticket $EM_Ticket
         */
        if (!is_a($EM_Bookings, 'EM_Bookings'))
            return $available_spaces;

        $spaces = $EM_Bookings->get_spaces();

        $relevant_booked_spaces = 0;
        foreach ($EM_Bookings->get_tickets() as $EM_Ticket) {
            if (isset($EM_Ticket->ticket_meta['event_spaces_irrelevant']) && $EM_Ticket->ticket_meta['event_spaces_irrelevant'] == 1) {
                $spaces += $EM_Ticket->get_spaces() - $EM_Ticket->get_booked_spaces();
                if (get_option('dbem_bookings_approval_reserved')) {
                    $spaces = $spaces - $EM_Ticket->get_pending_spaces();
                }
            }
        }

        foreach ($EM_Bookings->bookings as $EM_Booking)
            if ($EM_Booking->booking_status == 1 || (!get_option('dbem_bookings_approval') && $EM_Booking->booking_status == 0))
                foreach ($EM_Booking->get_tickets_bookings() as $EM_Ticket_Booking) {
                    $EM_Ticket = $EM_Ticket_Booking->get_ticket();
                    if(isset($EM_Ticket->ticket_meta['event_spaces_irrelevant']))
                        $event_spaces_irrelevant = $EM_Ticket->ticket_meta['event_spaces_irrelevant'];
                    else
                        $event_spaces_irrelevant = get_post_meta($EM_Ticket->get_event()->post_id, 'ticket' . $EM_Ticket->ticket_id . '_event_spaces_irrelevant', true);

                    if (!$event_spaces_irrelevant)
                        $relevant_booked_spaces += $EM_Ticket_Booking->get_spaces();
                }

        $available_spaces = $spaces - $relevant_booked_spaces;
        if (get_option('dbem_bookings_approval_reserved')) {

            if( get_option('dbem_bookings_approval') == 0 )
                $relevant_pending_spaces = apply_filters('em_bookings_get_pending_spaces', 0, $this);
            else {
                $relevant_pending_spaces = 0;
                foreach ($EM_Bookings->bookings as $EM_Booking)
                    if ($EM_Booking->booking_status == 0)
                        foreach ($EM_Booking->get_tickets_bookings() as $EM_Ticket_Booking) {
                            $EM_Ticket = $EM_Ticket_Booking->get_ticket();
                            if(isset($EM_Ticket->ticket_meta['event_spaces_irrelevant']))
                                $event_spaces_irrelevant = $EM_Ticket->ticket_meta['event_spaces_irrelevant'];
                            else
                                $event_spaces_irrelevant = get_post_meta($EM_Ticket->get_event()->post_id, 'ticket' . $EM_Ticket->ticket_id . '_event_spaces_irrelevant', true);

                            if (!$event_spaces_irrelevant)
                                $relevant_pending_spaces += $EM_Ticket_Booking->get_spaces();
                        }
            }

            $available_spaces -= $relevant_pending_spaces;
        }
        return $available_spaces;
    }

    /**
     * @param $available_spaces
     * @param EM_Bookings $EM_Bookings
     * @return int
     */
    public function get_pending_irrelevant_spaces($EM_Bookings)
    {
        $irrelevant_pending_spaces = 0;
        foreach ($EM_Bookings->bookings as $EM_Booking) {
            if ($EM_Booking->booking_status == 0){
                foreach ($EM_Booking->get_tickets_bookings() as $EM_Ticket_Booking) {
                    $EM_Ticket = $EM_Ticket_Booking->get_ticket();
                    if (isset($EM_Ticket->ticket_meta['event_spaces_irrelevant']))
                        $event_spaces_irrelevant = $EM_Ticket->ticket_meta['event_spaces_irrelevant'];
                    else
                        $event_spaces_irrelevant = get_post_meta($EM_Ticket->get_event()->post_id, 'ticket' . $EM_Ticket->ticket_id . '_event_spaces_irrelevant', true);

                    if ($event_spaces_irrelevant)
                        $irrelevant_pending_spaces += $EM_Ticket_Booking->get_spaces();
                }
            }
        }
        return $irrelevant_pending_spaces;
    }


    /**
     * @param $available_spaces
     * @param EM_Bookings $EM_Bookings
     * @return int
     */
    public function get_booked_irrelevant_spaces($EM_Bookings)
    {
        /**
         * @var EM_Booking $EM_Booking
         * @var EM_Ticket $EM_Ticket
         */
        if (!is_a($EM_Bookings, 'EM_Bookings'))
            return 0;

        $irrelevant_booked_spaces = 0;
        foreach ($EM_Bookings->bookings as $EM_Booking) {
            if ($EM_Booking->booking_status == 1 || (!get_option('dbem_bookings_approval') && $EM_Booking->booking_status == 0)) {
                foreach ($EM_Booking->get_tickets_bookings() as $EM_Ticket_Booking) {
                    $EM_Ticket = $EM_Ticket_Booking->get_ticket();
                    if(isset($EM_Ticket->ticket_meta['event_spaces_irrelevant']))
                        $event_spaces_irrelevant = $EM_Ticket->ticket_meta['event_spaces_irrelevant'];
                    else
                        $event_spaces_irrelevant = get_post_meta($EM_Ticket->get_event()->post_id, 'ticket' . $EM_Ticket->ticket_id . '_event_spaces_irrelevant', true);

                    if ($event_spaces_irrelevant)
                        $irrelevant_booked_spaces += $EM_Ticket_Booking->get_spaces();
                }
            }
        }
        if (get_option('dbem_bookings_approval_reserved')) {
            $irrelevant_pending_spaces = $this->get_pending_irrelevant_spaces($EM_Bookings);
            $irrelevant_booked_spaces -= $irrelevant_pending_spaces;
        }
        return $irrelevant_booked_spaces;
    }

    /**
     * @param $available_spaces
     * @param EM_Ticket $EM_Ticket
     * @return int
     */
    public function get_ticket_relevant_spaces($available_spaces, $EM_Ticket){
        $event_available_spaces = $EM_Ticket->get_event()->get_bookings()->get_available_spaces();
        $ticket_available_spaces = $EM_Ticket->get_spaces() - $EM_Ticket->get_booked_spaces();


        $irrelevantspaces = 0;
        foreach ($EM_Ticket->get_event()->get_bookings()->get_tickets() as $Tmp_Ticket) {
            if (isset($Tmp_Ticket->ticket_meta['event_spaces_irrelevant']) && $Tmp_Ticket->ticket_meta['event_spaces_irrelevant'] == 1)
                $irrelevantspaces += $Tmp_Ticket->get_spaces() - $Tmp_Ticket->get_booked_spaces();
        }

        if( get_option('dbem_bookings_approval_reserved')){
            $ticket_available_spaces = $ticket_available_spaces - $EM_Ticket->get_pending_spaces();
        }
        if(isset($EM_Ticket->ticket_meta['event_spaces_irrelevant']))
            $event_spaces_irrelevant = $EM_Ticket->ticket_meta['event_spaces_irrelevant'];
        else
            $event_spaces_irrelevant = get_post_meta($EM_Ticket->get_event()->post_id, 'ticket' . $EM_Ticket->ticket_id . '_event_spaces_irrelevant', true);
        if ($event_spaces_irrelevant) {
            $event_available_spaces  += $irrelevantspaces;
            $return = ($ticket_available_spaces <= $event_available_spaces) ? $ticket_available_spaces : $event_available_spaces;
        } else {
            $event_available_spaces -= $irrelevantspaces;
            $return = ($ticket_available_spaces <= $event_available_spaces) ? $ticket_available_spaces : $event_available_spaces;
        }

        return $return;
    }


    /**
     * Returns EM_Tickets object with available tickets
     * @param boolean $include_member_tickets - if set to true, member-ony tickets will be considered available even if logged out
     * @return EM_Tickets
     */
    function get_available_tickets( $EM_Tickets,  $EM_Booking){
        $tickets = array();
        $tmp = $EM_Booking->get_tickets();
        foreach ($EM_Booking->get_tickets() as $EM_Ticket){
            /* @var $EM_Ticket EM_Ticket */
            if( $EM_Ticket->is_available() ){
                //within time range
                if( $EM_Ticket->get_available_spaces() > 0 ){
                    $tickets[] = $EM_Ticket;
                }
            }
        }
        $EM_Tickets = new EM_Tickets($tickets);
        return $EM_Tickets;
    }

    /**
     * ATTENTION: hook "em_booking_get_spaces" is not only used by EM_Booking::get_spaces. Also for example in EM_Tickets_Bookings::get_spaces I think its a bug in events_manager
     *
     * @param $spaces
     * @param $obj
     * @return int
     */
    public function get_booking_spaces($spaces, $obj)
    {
        switch (get_class($obj)){
//            case 'EM_Ticket_Booking':
//                /** @var EM_Ticket_Booking $obj */
//                $EM_Ticket = $obj->get_ticket();
//                if(isset($EM_Ticket->ticket_meta['event_spaces_irrelevant']))
//                    $event_spaces_irrelevant = $EM_Ticket->ticket_meta['event_spaces_irrelevant'];
//                else
//                    $event_spaces_irrelevant = get_post_meta($EM_Ticket->get_event()->post_id, 'ticket' . $EM_Ticket->ticket_id . '_event_spaces_irrelevant', true);
//
//                if($event_spaces_irrelevant)
//                    $spaces = 0;
//                break;
            case 'EM_Tickets_Bookings':
                /** @var EM_Tickets_Bookings $obj */
                $relevant_booked_spaces = 0;
                foreach($obj->tickets_bookings as $EM_Ticket_Booking){
                    /* @var $EM_Ticket_Booking EM_Ticket_Booking */
                    $EM_Ticket = $EM_Ticket_Booking->get_ticket();
                    if(isset($EM_Ticket->ticket_meta['event_spaces_irrelevant']))
                        $event_spaces_irrelevant = $EM_Ticket->ticket_meta['event_spaces_irrelevant'];
                    else
                        $event_spaces_irrelevant = get_post_meta($EM_Ticket->get_event()->post_id, 'ticket' . $EM_Ticket->ticket_id . '_event_spaces_irrelevant', true);

                    if(!$event_spaces_irrelevant)
                        $relevant_booked_spaces += $EM_Ticket_Booking->get_spaces();
                }
                break;
//            case 'EM_Bookings':
//                /** @var EM_Bookings $obj */
//                $relevant_booked_spaces = 0;
//                foreach($obj->tickets as $EM_Ticket){
//                    if(isset($EM_Ticket->ticket_meta['event_spaces_irrelevant']))
//                        $event_spaces_irrelevant = $EM_Ticket->ticket_meta['event_spaces_irrelevant'];
//                    else
//                        $event_spaces_irrelevant = get_post_meta($EM_Ticket->get_event()->post_id, 'ticket' . $EM_Ticket->ticket_id . '_event_spaces_irrelevant', true);
//
//                    if(!$event_spaces_irrelevant)
//                        $relevant_booked_spaces += $EM_Ticket->get_spaces();
//                }
//                //check overall events cap
//                if(!empty($obj->get_event()->event_spaces) && $obj->get_event()->event_spaces < $spaces) $relevant_booked_spaces = $obj->get_event()->event_spaces;
//                break;
            case 'EM_Booking':
                $relevant_booked_spaces = 0;
                /** @var EM_Booking $obj */
                foreach ($obj->get_tickets_bookings() as $EM_Ticket_Booking) {
                    /** @var EM_Ticket $EM_Ticket */
                    $EM_Ticket = $EM_Ticket_Booking->get_ticket();
                    if (isset($EM_Ticket->ticket_meta['event_spaces_irrelevant']))
                        $event_spaces_irrelevant = $EM_Ticket->ticket_meta['event_spaces_irrelevant'];
                    else
                        $event_spaces_irrelevant = get_post_meta($EM_Ticket->get_event()->post_id, 'ticket' . $EM_Ticket->ticket_id . '_event_spaces_irrelevant', true);

                    if (!$event_spaces_irrelevant)
                        $relevant_booked_spaces += $EM_Ticket_Booking->get_spaces();
                }
                break;
        }

        $return = isset($relevant_booked_spaces) && $relevant_booked_spaces < $spaces ? $relevant_booked_spaces : $spaces;
        return $return;
    }


    /**
     * @param $result
     * @param EM_Booking $EM_Booking
     * @return bool
     */
    public function em_booking_validate_without_tickets($result, $EM_Booking)
    {
        if(!$result){

			//return false if there are more errors than one
			if(count($EM_Booking->errors) > 1)
				return false;

            //check if there is the min space error
            foreach ($EM_Booking->errors as $k => $error){
                switch ($error){
                    case get_option('dbem_booking_feedback_min_space'):
                        //                case get_option('dbem_booking_feedback_full'):
                        //                case sprintf(get_option('dbem_booking_feedback_spaces_limit'), $EM_Booking->get_event()->event_rsvp_spaces):
                        unset($EM_Booking->errors[$k]);
                        break;
                }
            }

            //return false if something other went wrong
            if(!empty($EM_Booking->errors))
                return false;

            // tickets bookings info
			$EM_Tickets_Bookings = $EM_Booking->get_tickets_bookings();
            if( count($EM_Tickets_Bookings->tickets_bookings) > 0 || $EM_Booking->booking_spaces > 0 ){
                $result = true;
            } else {
				$EM_Booking->add_error(get_option('dbem_booking_feedback_min_space'));
				return false;
            }

            //is there enough space overall?
            if( $EM_Booking->get_event()->get_bookings()->get_available_spaces() < $EM_Booking->get_spaces() ){
                $result = false;
                $EM_Booking->add_error(get_option('dbem_booking_feedback_full'));
            }
            //can we book this amount of spaces at once?
            if( $EM_Booking->get_event()->event_rsvp_spaces > 0 && $EM_Booking->get_spaces() > $EM_Booking->get_event()->event_rsvp_spaces ){
                $result = false;
                $EM_Booking->add_error( sprintf(get_option('dbem_booking_feedback_spaces_limit'), $EM_Booking->get_event()->event_rsvp_spaces));
            }
        }
        return $result;
    }

    /**
     * @return string
     */
    public function locate_template($template_name, $load, $the_args = array() )
    {
        $tmp = plugin_dir_path(dirname(__FILE__)) .'plugins/events-manager/templates/'.$load;
        if (file_exists($tmp))
            $template_name = $tmp;
        return $template_name;
    }


    /**
     * REQUIRES v4-wp1crm-extensions
     *
     * @param $message
     * @param bool $is_error
     */
    public function add_admin_notice($message, $is_error = false)
    {
        if (!session_id()) {
            session_start();
        }

        $msg_exists = false;
        if (isset($_SESSION['wpwh_error']))
            foreach ($_SESSION['wpwh_error'] as $error)
                if ($message == $error['msg'])
                    $msg_exists = true;

        if (!$msg_exists)
            $_SESSION['wpwh_error'][] = [
                'msg' => $message,
                'is_error' => $is_error
            ];
    }

	public function get_default_role()
	{
		return
			isset($this->wpwh_options['default_role']) && get_role($this->wpwh_options['default_role'])
				? $this->wpwh_options['default_role']
				: 'subscriber';
    }

    public function get_register_fields()
    {
        return $this->register_fields;
    }

    public function get_my_account_fields()
    {
        $my_account_fields = $this->register_fields;

        $my_account_fields['billing_title']['editable'] = false;
        $my_account_fields['billing_first_name']['editable'] = false;
        $my_account_fields['billing_last_name']['editable'] = false;

		$wp1crm = get_option('wp1crm');
		if($wp1crm['name_readonly']){
			$my_account_fields['billing_first_name']['editable'] = false;
			$my_account_fields['billing_last_name']['editable'] = false;
		}
		if($wp1crm['company_readonly']){
			$my_account_fields['billing_company']['editable'] = false;
		}
		if($wp1crm['email_readonly']){
			$my_account_fields['email']['editable'] = false;
		}

        return apply_filters('v4-wp-webhook/my-account-fields', $my_account_fields);
    }

    private function set_register_fields()
    {
		$wp1crm = get_option('wp1crm');
		$this->register_fields = [];
		$this->register_fields['billing_title'] = [
			'name' => 'Salutation',
			'validation_type' => 'int',
			'required' => true,
			'type' => 'select',
			'options' => [
				'1' => __('Mr.', 'v4-wp-webhook'),
				'2' => __('Ms.', 'v4-wp-webhook'),
				'3' => __('diverse', 'v4-wp-webhook')
			],
			'classes' => [
				'full'
			]
		];
		$this->register_fields['billing_first_name'] = [
			'name' => 'First Name',
			'validation_type' => 'string',
			'required' => true,
			'type' => 'text'
		];
		$this->register_fields['billing_last_name'] = [
			'name' => 'Surname',
			'validation_type' => 'string',
			'required' => true,
			'type' => 'text'
		];
		$company_required = 'required' === get_option('woocommerce_checkout_company_field', 'optional');
		$this->register_fields['billing_company'] = [
			'name' => 'Company',
			'validation_type' => 'string',
			'required' => $company_required,
			'type' => 'text',
			'classes' => [
				'full'
			]
		];
		if(isset($wp1crm['b2b']) && $wp1crm['b2b']){
			$this->register_fields['tax_information'] = [
				'name' => 'USt.ID.',
				'validation_type' => 'string',
//				'required' => true,
				'type' => 'text',
				'classes' => [
					'full'
				]
			];
		}

		$this->register_fields['billing_country'] = [
			'name' => 'Country',
			'validation_type' => 'string',
			'required' => true,
			'type' => 'select',
			'options' => $this->get_countries(),
			'default' => 'DE',
			'classes' => [
				'full'
			]
		];
		$this->register_fields['billing_address_1'] = [
			'name' => 'Street',
			'validation_type' => 'string',
			'required' => true,
			'type' => 'text',
			'classes' => [
				'full'
			]
		];
		$this->register_fields['billing_postcode'] = [
			'name' => 'Postcode',
			'validation_type' => 'string',
			'required' => true,
			'type' => 'text'
		];
		$this->register_fields['billing_city'] = [
			'name' => 'City',
			'validation_type' => 'string',
			'required' => true,
			'type' => 'text'
		];

//		if (function_exists('WC'))
//			$this->register_fields['billing_state'] = [
//				'name' => 'State',
//				'validation_type' => 'string',
//				'required' => false,
//				'type' => 'select',
//				'options' => $this->get_states(),
//				'classes' => [
//					'full'
//				]
//			];
		$phone_required = 'required' === get_option('woocommerce_checkout_phone_field', 'required');
		$this->register_fields['billing_phone'] = [
			'name' => 'Phone',
			'validation_type' => 'string',
			'required' => $phone_required,
			'type' => 'text',
			'classes' => [
				'full'
			]
		];
		$this->register_fields['email'] = [
			'name' => 'Email',
			'validation_type' => 'email',
			'required' => true,
			'type' => 'email',
			'classes' => [
				'full'
			]
		];
        // honeypot
		$this->register_fields['street'] = [
			'name' => 'Street',
			'validation_type' => 'honeypot',
			'required' => true,
			'type' => 'text',
			'classes' => [
				'full',
				'v4hp',
			]
		];

        $recaptcha_site_key = $this->wpwh_options['recaptcha_site_key'] ?? false;
        $recaptcha_secret_key = $this->wpwh_options['recaptcha_secret_key'] ?? false;
		if (!is_user_logged_in() && !empty($recaptcha_site_key) && !empty($recaptcha_secret_key)) {
            $this->register_fields['recaptcha'] = [
                'name' => 'reCaptcha',
                'type' => 'custom',
                'validation_type' => 'custom',
                'meta' => [
                    'site_key' => $recaptcha_site_key,
                    'secret_key' => $recaptcha_secret_key,
                ]
            ];

            add_action('v4wpwh_register_field_html_recaptcha', [$this, 'template_recaptcha'], 10, 0);
            add_filter('v4wpwh_register_field_validation_recaptcha', [$this, 'validate_recaptcha'], 10, 0);
            //add_filter('v4wpwh_register_field_validation_msg_recaptcha', [$this, 'validate_recaptcha_msg'], 10, 1); // no need

            //EMP

            add_action('booking_form_after_user_details', [$this, 'template_recaptcha'], 5, 0);
            add_action('em_checkout_form_footer', [$this, 'template_recaptcha'], 5, 0);
            if (empty(apply_filters('v4_remove_recaptcha_from_em_booking', null))) {
                add_action('em_booking_form_footer', [$this, 'template_recaptcha'], 5, 0);
            }

            //WC
            add_action('woocommerce_after_checkout_billing_form', [$this, 'template_recaptcha'], 10, 1);
            if (empty(apply_filters('v4_remove_recaptcha_from_em_booking', null))) {
                add_action('woocommerce_after_checkout_validation', [$this, 'wc_validate_recaptcha'], 10, 2);
            }

        }

        $this->register_fields = apply_filters('v4-wp-webhook/register-fields', $this->register_fields);
    }

    public function get_countries()
    {
        return $this->countries;
    }

    private function set_countries()
    {
        if (function_exists('WC'))
            $countries = WC()->countries->get_countries();
        else if (function_exists('em_get_countries'))
            $countries = em_get_countries();
        else
            $countries = include __DIR__ . '/countries.php';

        uasort($countries, function ($a, $b) {
			if ($a == $b) {
				return 0;
			}
			return ($a < $b) ? -1 : 1;
        });

        $this->countries = apply_filters('v4-wp-webhook/countries', $countries);
    }

	public function get_states()
	{
		return $this->states;
	}

	private function set_states()
	{
		$states = [];

		if (function_exists('WC'))
			$states = WC()->countries->get_states();

		uasort($states, function ($a, $b) {
			if ($a == $b) {
				return 0;
			}
			return ($a < $b) ? -1 : 1;
		});

		$this->states = apply_filters('v4-wp-webhook/states', $states);
	}

    /**
     * register action
     */
    public function wpwh_my_account_admin_post()
    {
		// clean ob because of eventual warnings. We don't want them in js
		ob_clean();
		$existing_user_id = get_current_user_id();
		$user = [];
		$errors = [];
		$result = [
			'status' => 0,
			'msg' => '',
			'errors' => [],
		];

		$user['id'] = $existing_user_id;
		$user['meta_data'][] = [
			'key' => 'crm_id',
			'value' => get_user_meta($existing_user_id, 'crm_id', true)
		];
		if(!$existing_user_id) {
			$errors[] = __('Please log in again.');
		}

		$this->validate_my_account_fields($user, $errors);

        if (count($errors) == 0) {
            try {
//                $this->wpwh_update_user($user);
//                $crm_result = $this->wpwh_send_user($user);
				$v4Customer = new v4Customer($user);
				$crm_result = $v4Customer->save();

                if ($crm_result) {
                    $result['status'] = 200;
                    $result['msg'] = __("Success! Your personal data has been changed.", "v4-wp-webhook");
                }
            } catch (Exception $e) {
                $result['status'] = $e->getCode();
                if ($e->getCode() == 400) {
                    $result['msg'] = __($e->getMessage(), "v4-wp-webhook");
                } else {
                    error_log('v4-wp-webhook (' . $result['status'] . '): ' . $e->getMessage());
                    $result['msg'] = __("Oops, that should not have happened... Please try again later or contact the administrator of this website.", "v4-wp-webhook") . " ({$e->getCode()})";
                }
            }
        } else {
            $result['status'] = 400;
            $result['msg'] = __("Please check your input and try again.", "v4-wp-webhook");
            $result['errors'] = $errors;
        }

        echo json_encode($result);
        die();
    }
	public function validate_my_account_fields(&$user, &$errors)
	{
		foreach ($this->get_my_account_fields() as $k => $var) {
			if (isset($var['editable']) && $var['editable'] === false) {
				switch ($k) {
					case 'email':
						$current_user = wp_get_current_user();
						$value = $current_user->data->user_email;
						break;
					default:
						$value = $k ? v4Customer::maybe_unserialize(get_user_meta(get_current_user_id(), $k, true)) : '';
						break;
				}
				$user[$k] = $value;
			} else {
				$this->validate_field($k, $var, $user, $errors);
			}
		}
	}

    /**
     * register action
     */
    public function wpwh_register_admin_post_nopriv()
    {
		// clean ob because of eventual warnings. We don't want them in js
		ob_clean();

		$my_current_lang = apply_filters( 'wpml_current_language', NULL );
		if(!empty($my_current_lang)){
			global $sitepress;
			$sitepress->switch_lang($my_current_lang);
		}

        if (is_user_logged_in()) {
            $result = [
                'status' => 400,
                'msg' => __("Please logout before register.", "v4-wp-webhook"),
                'errors' => [],
            ];
            echo json_encode($result);
            die();
        }

        $user = [];
        $errors = [];

		$this->validate_consent($errors);

        $this->validate_register_fields($user, $errors);

        if (!empty($user['email']) && email_exists($user['email'])) {
            $result = [
                'status' => 400,
                'msg' => get_option('dbem_booking_feedback_email_exists'),
                'errors' => [],
            ];
            echo json_encode($result);
            die();
        }


        $result = [
            'status' => 0,
            'msg' => '',
            'errors' => [],
        ];
        if (count($errors) == 0) {
            try {
//                $crm_result = $this->wpwh_send_user($user);
				$v4Customer = new v4Customer($user);
				$crm_result = $v4Customer->save();
				if ($crm_result) {
					$result['status'] = 200;
					$result['msg'] = __("Success! You will receive an e-mail with your login information soon.", "v4-wp-webhook");
					if (!empty($_POST['instant_login']) && $_POST['instant_login'] == '1') {
						$result = $this->instant_login($user, $crm_result, $result);
					}
				}
			} catch (Exception $e) {
                $result['status'] = $e->getCode();
                if ($e->getCode() == 400) {
                    $result['msg'] = __($e->getMessage(), "v4-wp-webhook");
                } else {
                    error_log('v4-wp-webhook (' . $result['status'] . '): ' . $e->getMessage());
                    $result['msg'] = __("Oops, that should not have happened... Please try again later or contact the administrator of this website.", "v4-wp-webhook") . " ({$e->getCode()})";
                }
            }
        } else {
            $result['status'] = 400;
            $result['msg'] = __("Please check your input and try again.", "v4-wp-webhook");
            $result['errors'] = $errors;
        }

        echo json_encode($result);
        die();
    }

	public function validate_consent(&$errors)
	{
		if (empty($_POST['consent']) || $_POST['consent'] !== "1") {
			if (!empty($this->wpwh_options['consent_privacy_policy']) && !empty($this->wpwh_options['consent_terms_of_participation']))
				$errors['consent'] = sprintf(
					__('You have to agree our <a target="_blank" href="%s">privacy policy</a> and <a target="_blank" href="%s">terms and conditions</a>.', 'v4-wp-webhook'),
					$this->wpwh_options['consent_privacy_policy'],
					$this->wpwh_options['consent_terms_of_participation']
				);
			else if (!empty($this->wpwh_options['consent_privacy_policy']))
				$errors['consent'] = sprintf(
					__('You have to agree our <a target="_blank" href="%s">privacy policy</a>.', 'v4-wp-webhook'),
					$this->wpwh_options['consent_privacy_policy']
				);
			else if (!empty($this->wpwh_options['consent_terms_of_participation']))
				$errors['consent'] = sprintf(
					__('You have to agree our privacy policy and <a target="_blank" href="%s">terms and conditions</a>.', 'v4-wp-webhook'),
					$this->wpwh_options['consent_terms_of_participation']
				);
			else
				$errors['consent'] = __('You have to agree our privacy policy.', 'v4-wp-webhook');
		}
	}

	public function validate_register_fields(&$user, &$errors)
	{
		foreach ($this->get_register_fields() as $k => $var) {
			$this->validate_field($k, $var, $user, $errors);
		}
	}

	private function validate_field($post_key, $field, &$user, &$errors)
	{
		$error_msg = sprintf(esc_html__('Please enter a valid %1$s', 'v4-wp-webhook'), __($field['name'], 'v4-wp-webhook'));
		if(!isset($_POST[$post_key]) && isset($_POST['billing_' . $post_key]))
			$_POST[$post_key] = $_POST['billing_' . $post_key];

		if(empty($field['validation_type']))
			$field['validation_type'] = $field['type'];

		switch ($field['validation_type']) {
			case 'int':
				if (!empty($_POST[$post_key]) && filter_var($_POST[$post_key], FILTER_VALIDATE_INT) != false)
					$user[$post_key] = $_POST[$post_key];
				else if (isset($field['required']) && $field['required'] || !empty($_POST[$post_key]))
					$errors[$post_key] = $error_msg;
				break;
			case 'email':
				if (!empty($_POST[$post_key]) && filter_var($_POST[$post_key], FILTER_VALIDATE_EMAIL) != false)
					$user[$post_key] = $_POST[$post_key];
				else if (isset($field['required']) && $field['required'] || !empty($_POST[$post_key]))
					$errors[$post_key] = $error_msg;
				break;
			case 'select':
				if (
					(!empty($_POST[$post_key]) && $field['multiple'] && is_array($_POST[$post_key]) && count(array_intersect($_POST[$post_key], array_keys($field['options']))) == count($_POST[$post_key]))
					|| !empty($_POST[$post_key]) && in_array($_POST[$post_key], array_keys($field['options']))
				)
					$user[$post_key] = $_POST[$post_key];
				else if (isset($field['required']) && $field['required'] || !empty($_POST[$post_key]))
					$errors[$post_key] = $error_msg;
				break;
			case 'date':
				if (!empty($_POST[$post_key]) && date('d.m.Y', strtotime($_POST[$post_key])) !== '01.01.1970')
					$user[$post_key] = date('Y-m-d', strtotime($_POST[$post_key]));
				else if (isset($field['required']) && $field['required'] || !empty($_POST[$post_key]))
					$errors[$post_key] = $error_msg;
				break;
			case 'custom':
				$is_valid = apply_filters('v4wpwh_register_field_validation_' . $post_key, true, $_POST[$post_key]??null, $field);
				if ($is_valid)
					$user[$post_key] = $_POST[$post_key]??true;
				else
					$errors[$post_key] = apply_filters('v4wpwh_register_field_validation_msg_' . $post_key, $error_msg);
				break;
			case 'honeypot':
				if (!empty($_POST[$post_key]))
					$errors[$post_key] = $error_msg;
				break;
			default:
				$val = !empty($_POST[$post_key]) ? filter_var($_POST[$post_key], FILTER_SANITIZE_STRING) : false;
				if ($val)
					$user[$post_key] = $val;
				else if (isset($field['required']) && $field['required'] || !empty($_POST[$post_key]))
					$errors[$post_key] = $error_msg;
				break;
		}
	}

    public function instant_login($user, $crm_result, $result)
    {
        //check status
        if ($result['status'] !== 200)
            return $result;

        //is instant login allowed?
        if (!isset($this->wpwh_options['allow_instant_login']) || !$this->wpwh_options['allow_instant_login'])
            return $result;

        //decode json
        $crm_result_arr = json_decode($crm_result, true);
        if (!is_array($crm_result_arr)) {
            $result['code'] = 500;
            return $result;
        }

        //check id only one Contact
        if (empty($crm_result_arr['Contact']) || count($crm_result_arr['Contact']) !== 1) {
            $result['code'] = 500;
            return $result;
        }

        //check if v4oauth is installed
        if (!file_exists(__DIR__ . '/../../v4-oauth/User.php')) {
            $result['code'] = 500;
            return $result;
        }
        require_once __DIR__ . '/../../v4-oauth/User.php';

        /*
         * do login
         */
        $v4oAuthUser = new \v4\wp\plugins\oauth\User(false);
        $v4oAuthUser->me = [
            "id" => $crm_result_arr['Contact'][0]['id'],
            "email" => $user['email'],
            "name" => $user['first_name'] . ' ' . $user['surname'],
            "first_name" => $user['first_name'],
            "last_name" => $user['surname'],
            "role" => $this->get_default_role(),
            "billing_address" => [
                "company" => $user['company'],
                "contact" => $user['first_name'] . ' ' . $user['surname'],
                "street" => $user['street'],
                "city" => $user['city'],
                "state" => $user['state'],
                "postalcode" => $user['postcode'],
                "country" => $user['country']
            ],
            "shipping_address" => [
                "company" => null,
                "contact" => "",
                "street" => "",
                "city" => "",
                "state" => "",
                "postalcode" => "",
                "country" => ""
            ],
            "salutation" => $user['salutation'],
            "phone" => $user['phone'],
            "birthdate" => '',
        ];
        try {
            $v4oAuthUser->login();
            $result['reload'] = true;
        } catch (Exception $e) {
            $result['status'] = $e->getCode();
            $result['msg'] = $e->getMessage();
        }

        return $result;
    }

    /**
     * user update webhook
     *
     * @param $user
     * @return array
     * @throws Exception
     */
    public function wpwh_update_user($user)
    {
        $user_id = get_current_user_id();
        $result = [];

        if ($user_id) {

            /*
             * update user data
             */
            $result['update']['user'] = wp_update_user(array(
                'ID' => $user_id,
                'user_email' => $user['email']
            ));

            /*
             * update user meta data
             */
            $result['update']['first_name'] = update_user_meta($user_id, 'first_name', $user['first_name'] ?? '');
            $result['update']['last_name'] = update_user_meta($user_id, 'last_name', $user['surname'] ?? '');

            $result['update']['billing_birthdate'] = update_user_meta($user_id, 'billing_birthday', $user['billing_birthday'] ?? '');
            $result['update']['billing_first_name'] = update_user_meta($user_id, 'billing_first_name', $user['first_name'] ?? '');
            $result['update']['billing_last_name'] = update_user_meta($user_id, 'billing_last_name', $user['surname'] ?? '');


            $result['update']['billing_address_1'] = update_user_meta($user_id, 'billing_address_1', $user['street'] ?? '');

            $result['update']['billing_city'] = update_user_meta($user_id, 'billing_city', $user['city'] ?? '');
            $result['update']['billing_postcode'] = update_user_meta($user_id, 'billing_postcode', $user['postcode'] ?? '');
            $result['update']['billing_country'] = update_user_meta($user_id, 'billing_country', $user['country'] ?? '');
            $result['update']['billing_state'] = update_user_meta($user_id, 'billing_state', $user['state'] ?? '');

            $result['update']['billing_company'] = update_user_meta($user_id, 'billing_company', $user['company'] ?? '');
            $result['update']['billing_email'] = update_user_meta($user_id, 'billing_email', $user['email'] ?? '');
            $result['update']['billing_title'] = update_user_meta($user_id, 'billing_title', $user['salutation'] ?? '');
            $result['update']['billing_salutation'] = update_user_meta($user_id, 'billing_salutation', $user['salutation'] ?? '');
            $result['update']['billing_phone'] = update_user_meta($user_id, 'billing_phone', str_replace('/', ' ', $user['phone'] ?? ''));

            /* events manager */
            update_user_meta($user_id, 'dbem_address', $user['street'] ?? '');
            update_user_meta($user_id, 'dbem_city', $user['city'] ?? '');
            update_user_meta($user_id, 'dbem_zip', $user['postcode'] ?? '');
            update_user_meta($user_id, 'dbem_country', $user['country'] ?? '');
            update_user_meta($user_id, 'dbem_state', $user['state'] ?? '');


            //WooCommerce Customer update
            if (function_exists('WC') && WC()->customer !== null) {
                WC()->customer->set_first_name($user['first_name'] ?? '');
                WC()->customer->set_last_name($user['surname'] ?? '');
                WC()->customer->set_billing_company($user['company'] ?? '');
                WC()->customer->set_billing_country($user['country'] ?? '');
                WC()->customer->set_billing_state($user['state'] ?? '');
                WC()->customer->set_billing_address_1($user['street'] ?? '');
                WC()->customer->set_billing_city($user['city'] ?? '');
                WC()->customer->set_billing_postcode($user['postcode'] ?? '');
                WC()->customer->set_billing_phone($user['phone'] ?? '');
                WC()->customer->set_email($user['email'] ?? '');
            }


            //put rest in meta_data
            unset(
                $user['email'],
                $user['first_name'],
                $user['surname'],
                $user['first_name'],
                $user['surname'],
                $user['company'],
                $user['street'],
                $user['city'],
                $user['postcode'],
                $user['country'],
                $user['state'],
                $user['email'],
                $user['phone'],
                $user['salutation'],
                $user['phone_mobile'],
                $user['billing_birthday']
            );
            if (!empty($user) && is_array($user))
                foreach ($user as $k => $v) {
                    $result['update']['user_meta'][$k] = update_user_meta($user_id, $k, v4Customer::maybe_unserialize($v));
                }
        }


        return $result;
    }

    /**
     * user register webhook
     *
     * @param $user
     * @return boolean
     * @throws Exception
     */
    public function wpwh_send_user($user)
    {
		//prevent loops
		global $wpwh;
		if (isset($wpwh['do_not_send']) && $wpwh['do_not_send']) {
			return false;
		}

        $existing_user_id = get_current_user_id();

		if(empty($user['billing_last_name']))
			return false;

		$data = $this->get_user_data($user, $existing_user_id);
		$v4Customer = new v4Customer($data);
		return $v4Customer->save();

//        //if legacy v3
//        if ($this->wpwh_options['use_legacy_v3']) {
//            //rename
//            $data['billing_address'] = $data['billing'];
//
//            //unset
//            unset(
//                $data['billing']
//            );
//
//            //prefix
//            $data = ['customer' => $data];
//        }
//        $json = json_encode($data);
//
//        $event = $existing_user_id ? 'updated' : 'created';
//
//        $result = v4WebHook::fire_webhook($json, 'customer', $event);
//
//        $result_arr = json_decode($result, true);
//        if ($existing_user_id && !empty($result_arr['Contact']) && count($result_arr['Contact']) === 1)
//            update_user_meta($existing_user_id, 'crm_id', $result_arr['Contact'][0]['id']);
//
//        return $result;
    }

	public function get_user_data($user, $existing_user_id = false)
	{
		global $wpdb;
		$data = [
			'email' => !empty($user['email']) ? $user['email'] : '',
			'first_name' => !empty($user['billing_first_name']) ? $user['billing_first_name'] : '',
			'last_name' => !empty($user['billing_last_name']) ? $user['billing_last_name'] : '',
			'role' => $this->get_default_role(),
			'billing' => [
				'first_name' => !empty($user['billing_first_name']) ? $user['billing_first_name'] : '',
				'last_name' => !empty($user['billing_last_name']) ? $user['billing_last_name'] : '',
				'company' => !empty($user['billing_company']) ? $user['billing_company'] : '',
			],
			'meta_data' => []
		];

		//billing
		if(isset($user['billing_address_1'])) $data['billing']['address_1'] = $user['billing_address_1'];
		if(isset($user['billing_city'])) $data['billing']['city'] = $user['billing_city'];
		if(isset($user['billing_postcode'])) $data['billing']['postcode'] = $user['billing_postcode'];
		if(isset($user['billing_country'])) $data['billing']['country'] = $user['billing_country'];
		if(isset($user['billing_state'])) $data['billing']['state'] = $user['billing_state'];
		if(isset($user['billing_email'])) $data['billing']['email'] = $user['billing_email'];
		if(isset($user['billing_phone'])) $data['billing']['phone'] = $user['billing_phone'];

		//meta
		if(isset($user['phone_mobile'])) $data['phone_mobile'] = $user['phone_mobile'];
		if(isset($user['billing_birthday'])) $data['billing_birthday'] = $user['billing_birthday'];
		if(isset($user['billing_title']))
			$data['meta_data'][] =
				[
					'key' => 'billing_title',
					'value' => !empty($user['billing_title']) ? $user['billing_title'] : '',
				];

		if ($existing_user_id) {
			$data['id'] = $existing_user_id;
			$data['meta_data'][] = [
				'key' => 'crm_id',
				'value' => get_user_meta($existing_user_id, 'crm_id', true)
			];
			// get all meta_data
			$person_meta = get_user_meta($existing_user_id, '', true);
			$meta_blacklist = [
				'first_name',
				'last_name',
				'session_tokens',
				$wpdb->prefix . 'capabilities',
				$wpdb->prefix . 'user_level',
				'_woocommerce_persistent_cart_1',
			];
			foreach ($person_meta as $k => $meta) {
				if(in_array($k, $meta_blacklist))
					continue;

				$meta = v4Customer::maybe_unserialize($meta);

				//if is single
				if(is_array($meta) && count($meta) === 1 && isset($meta[0]))
					$meta = $meta[0];

				$data['meta_data'][] =
					[
						'key' => $k,
						'value' => $meta
					];
			}
		}

		//put rest in meta_data
		unset(
			$user['email'],
			$user['billing_first_name'],
			$user['billing_last_name'],
			$user['billing_company'],
			$user['billing_address_1'],
			$user['billing_city'],
			$user['billing_postcode'],
			$user['billing_country'],
			$user['billing_state'],
			$user['billing_email'],
			$user['billing_phone'],
			$user['billing_title'],
			$user['phone_mobile'],
			$user['billing_birthday']
		);
		if (isset($data['meta_data']) && !empty($user) && is_array($user))
			foreach ($user as $k => $v) {
				$data['meta_data'][] = [
					'key' => $k,
					'value' => $v
				];
			}

		return $data;
	}

    /**
     * user register webhook
     *
     * @param bool $result
     * @param EM_Booking $EM_Booking
     * @return bool
     * @throws Exception
     */
    public function wpwh_send_booking($result, $EM_Booking)
    {
        //prevent loops
        global $wpwh;
		if (isset($wpwh['do_not_send']) && $wpwh['do_not_send']) {
            return $result;
        }

        // only proceed if sync_events isset
        if (!$this->wpwh_options['sync_event_bookings'])
            return $result;

		$EM_Event = $EM_Booking->get_event();

		//wpml: use original Event
		if(isset($EM_Event->event_translation) && $EM_Event->event_translation)
			$EM_Event = $EM_Event->get_parent();

		//don't sync recurring events (for now)
		if($EM_Event->is_recurring() || $EM_Event->is_recurrence())
			return true;

		// WooCommerce Support
		if(
			!empty($EM_Booking->booking_meta['woocommerce']['order_id']) &&
			empty($EM_Person->data->user_email) &&
			class_exists('\Events_Manager_WooCommerce\Orders')
		){
			$WC_Order = new WC_Order($EM_Booking->booking_meta['woocommerce']['order_id']);
			Events_Manager_WooCommerce\Orders::sync_order_billing_info($WC_Order, $EM_Booking);
		}
		$EM_Person = $EM_Booking->get_person();
		if($EM_Person->ID)
			$EM_Person = new EM_Person($EM_Person->ID);
		// return, if empty placeholder User is created. Will be saved a second time with data
		if ($EM_Booking->is_no_user() && empty($EM_Person->data->user_email)) {
			return true;
		}

        if ($result && $EM_Booking->validate()) {
            try {

				if (isset($this->wpwh_options['use_wpwh_register_fields']) && $this->wpwh_options['use_wpwh_register_fields']) {
					$user = [
						'id' => $EM_Person->ID
					];
					$errors = [];
					$this->validate_register_fields($user, $errors);
					if (count($errors) == 0) {
						try {
							$v4Customer = new v4Customer($user);
							$v4Customer->save();
						} catch (Exception $e) {
							if (!is_user_logged_in()) {
								if (is_admin()) {
									add_action('admin_notices', [$this, 'admin_notice__error_user']);
									error_log('v4-wp-webhook error (' . $e->getCode() . '): ' .  $e->getMessage());
								} else {
//									wp_delete_user($EM_Person->ID);
								}
							}
							throw $e;
						}
					} else {
						if(!is_user_logged_in()){
							if (is_admin()) {
								add_action('admin_notices', [$this, 'admin_notice__error_user']);
								error_log('v4-wp-webhook error: (400)' .  implode(', ', $errors));
							} else {
//								wp_delete_user($EM_Person->ID);
							}
						}
						$EM_Booking->add_error(implode(', ', $errors));
					}
				} else {
					$v4Customer = new v4Customer($EM_Person);
					$v4Customer->save();
				}

                if (is_a($EM_Booking, 'EM_Multiple_Booking')) {
                    /**
                     * @var EM_Multiple_Booking $EM_Booking
                     */
                    $data = $this->booking_webhook_data_multiple($EM_Booking);
                } else
                    $data = $this->booking_webhook_data($EM_Booking);

                // custom booking data
                $data = apply_filters('wpwh_booking_data', $data, $EM_Booking);

                //if legacy v3
                if ($this->wpwh_options['use_legacy_v3'])
                    $data = ['booking' => $data];

                $json = json_encode($data);
                $result = v4WebHook::fire_webhook($json, 'booking', 'created_updated');

                if (get_user_meta($EM_Person->ID, 'em_event_cancelled_post' . $EM_Event->post_id, true) == 1)
                    delete_user_meta($EM_Person->ID, 'em_event_cancelled_post' . $EM_Event->post_id);

                $result_arr = json_decode($result, true);
                if ($EM_Person->ID && !empty($result_arr['Contact']) && count($result_arr['Contact']) === 1)
                    update_user_meta($EM_Person->ID, 'crm_id', $result_arr['Contact'][0]['id']);

            } catch (Exception $e) {
                $result = false;
                $EM_Booking->add_error($e->getCode());
                $EM_Booking->add_error($e->getMessage());
            }

            if (!$result) {
                $wpwh['do_not_send'] = true;
                $EM_Booking->reject(false);
                unset($wpwh['do_not_send']);
                return false;
            }
        }

        return $result;
    }

	/**
	 * @param $result
	 * @param $EM_Booking
	 * @return bool|mixed
	 * @throws Exception
	 */
    public function wpwh_delete_booking($result, $EM_Booking)
    {
        if($result){
            //TODO delete?
            //set crm booking to canceled
//            $EM_Booking->booking_status = 3;
            $result = $this->wpwh_send_booking($result, $EM_Booking);
        }

        return $result;
    }

    /**
     * @param EM_Multiple_Booking $EM_Multiple_Booking
     * @return array
     */
    protected function booking_webhook_data_multiple(EM_Multiple_Booking $EM_Multiple_Booking): array
	{
        $meta = [];
        foreach ($EM_Multiple_Booking->booking_meta['booking'] as $k => $v) {
            $meta[] = [
                'key' => $k,
                'value' => $v
            ];
        }
        $data = [
            'type' => 'multiple',
            'bookings' => [],
            'meta_data' => $meta
        ];

//        $this->add_em_billing_address($data, $EM_Multiple_Booking);
//        $this->add_em_payment($data, $EM_Multiple_Booking);

        foreach ($EM_Multiple_Booking->bookings as $EM_Booking) {
            $booking_data = $this->booking_webhook_data($EM_Booking);
            $this->add_em_payment($booking_data, $EM_Multiple_Booking);
            $data['bookings'][] = $booking_data;
        }

        return $data;
    }

	/**
	 * @param EM_Booking $EM_Booking
	 * @return array
	 */
    public function booking_webhook_data($EM_Booking)
	{
        $EM_Event = $EM_Booking->get_event();

		//wpml: use original Event
		if(isset($EM_Event->event_translation) && $EM_Event->event_translation)
			$EM_Event = $EM_Event->get_parent();

		//don't sync recurring events (for now)
		if($EM_Event->is_recurring() || $EM_Event->is_recurrence())
			return [];

        $EM_Ticket_Bookings = $EM_Booking->get_tickets_bookings();
        $meta = [];
        if (!empty($EM_Booking->booking_meta['booking']))
            foreach ($EM_Booking->booking_meta['booking'] as $k => $v) {
                $meta[] = [
                    'key' => $k,
                    'value' => $v
                ];
            }

        $data = [
            'type' => 'simple',
            'id' => $EM_Booking->booking_id,
            'status' => $EM_Booking->booking_status,
            'price' => $EM_Booking->booking_price,
            'taxes' => $EM_Booking->booking_taxes,
            'tax_rate' => $EM_Booking->booking_tax_rate,
            'spaces' => $EM_Booking->booking_spaces,
            'event' => [
                'id' => $EM_Event->post_id //event_id nicht übers BE sichtbar für manuelle Anpassungen
            ],
            'meta_data' => $meta,
            'errors' => $EM_Booking->get_errors()
        ];

        if (isset($this->wpwh_options['use_legacy_v3']) && $this->wpwh_options['use_legacy_v3']) {
            $data['meta'] = $EM_Booking->booking_meta['booking'];
        }

        /**
         * Tickets
         *
         * @var EM_Ticket_Booking $EM_Ticket_Booking
         */
        if (!empty($EM_Ticket_Bookings)) {
            foreach ($EM_Ticket_Bookings->tickets_bookings as $EM_Ticket_Booking) {
                $EM_Ticket = !empty($EM_Ticket_Booking->ticket) ? $EM_Ticket_Booking->ticket : new EM_Ticket($EM_Ticket_Booking->ticket_id);
                $attendees = $EM_Booking->booking_meta['attendees'][$EM_Ticket->ticket_id] ?? [];

                $data['tickets_bookings'][] = [
                    'id' => $EM_Ticket_Booking->ticket_booking_id,
                    'price' => $EM_Ticket_Booking->ticket_booking_price,
                    'spaces' => $EM_Ticket_Booking->ticket_booking_spaces,
                    'ticket' => [
                        'id' => $EM_Ticket->ticket_id,
                        'name' => $EM_Ticket->ticket_name,
                        'description' => $EM_Ticket->ticket_description,
                        'price' => $EM_Ticket->ticket_price,
                        'start' => $EM_Ticket->start()->getTimestamp(),
                        'end' => $EM_Ticket->end()->getTimestamp(),
                        'spaces' => $EM_Ticket->ticket_spaces,
                        'spaces_available' => $EM_Ticket->ticket_spaces - $EM_Ticket->get_booked_spaces()
//                        'spaces_available' => $EM_Ticket_Booking->ticket->ticket_spaces - $EM_Ticket_Booking->ticket->get_booked_spaces() - $EM_Ticket_Booking->ticket_booking_spaces
                    ],
                    'attendees' => $attendees
                ];
            }
        }

        $this->add_em_billing_address($data, $EM_Booking);
        $this->add_em_user($data, $EM_Booking);
        $this->add_em_payment($data, $EM_Booking);

//        // prevent requests, if no user is submitted // made errors on anonymous bookings
//        if(empty($user_result))
//			throw new Exception('no user data submitted', 400);

        return $data;
    }

	/**
	 * @param $data
	 * @param EM_Booking $EM_Booking
	 * @throws Exception
	 */
    protected function add_em_user(&$data, EM_Booking $EM_Booking)
    {
		$EM_Person = $EM_Booking->get_person();
		$v4Customer = new v4Customer($EM_Person);
		$data['user'] = [
			'id' => $v4Customer->get_id(),
			'crm_id' => $v4Customer->get_crm_id()
		];
//		if($this->wpwh_options['use_wpwh_register_fields']){
//			$data['customer'] = $this->get_user_data($EM_Booking->booking_meta['registration'], $EM_Person->ID);
//		}

		/*
		$data['user'] = [
			'id' => $EM_Person->ID,
			'crm_id' => get_user_meta($EM_Person->ID, 'crm_id', true)
		];

		if($this->wpwh_options['use_wpwh_register_fields']){
			$data['customer'] = $this->get_user_data($EM_Booking->booking_meta['registration'], $EM_Person->ID);
			return true;
		}

		//sync customer fields if user is not logged in, or if is logged in and fields are shown
		if (!is_user_logged_in() || get_option('dbem_emp_booking_form_reg_show')) {
			// name
			$first_name = '';
			$name_arr = explode(' ', $EM_Person->data->user_name ?? $EM_Person->data->display_name);
			if (count($name_arr) > 1) {
				$first_name = $name_arr[0];
				unset($name_arr[0]);
			}
			$last_name = implode(' ', $name_arr);

			if(empty($last_name))
				return false;

			$data['customer'] = [
				'id' => $EM_Person->ID,
				'email' => $EM_Person->data->user_email,
				'first_name' => $first_name,
				'last_name' => $last_name,
				'role' => $this->get_default_role(),
				'billing' => [
					'first_name' => $first_name,
					'last_name' => $last_name,
					'email' => $EM_Person->data->user_email,
				]
			];


			// billing field mapping: em_field => crm_field
			$billing_fields = [
				'dbem_company' => 'company',
				'dbem_address' => 'address_1',
				'dbem_city' => 'city',
				'dbem_state' => 'state',
				'dbem_zip' => 'postcode',
				'dbem_country' => 'country',
				'dbem_phone' => 'phone',
				'dbem_title' => 'title'
			];
			if (is_plugin_active('events-manager-woocommerce/events-manager-woocommerce.php')) {
				$checkout = WC_Checkout::instance();
			}

			if($EM_Person->ID > 0){
				$data['customer']['meta_data'] = [
					[
						'key' => 'crm_id',
						'value' => get_user_meta($EM_Person->ID, 'crm_id', true)
					]
				];
				// get all meta_data
				$person_meta = get_user_meta($EM_Person->ID);
				//check if its woo checkout
				if (is_plugin_active('events-manager-woocommerce/events-manager-woocommerce.php') && isset($checkout)) {
					$checkout_fields = $checkout->get_checkout_fields();
					foreach ($checkout_fields['billing'] as $key => $checkout_field) {
						if(isset($person_meta[$key])){
							$prefix = 'billing_';
							if(strpos($key, $prefix) === 0){
								$billing_key = substr($key, strlen($prefix));
								$customer_billing_value = is_array($person_meta[$key]) ? implode(' ', $person_meta[$key]) : $person_meta[$key];
								$data['customer']['billing'][$billing_key] = $this->validate_input($customer_billing_value);
							}
						}
					}
				} else {
					if(!empty($EM_Person->custom_user_fields) && is_array($EM_Person->custom_user_fields))
						foreach ($EM_Person->custom_user_fields as $custom_user_field_key => $custom_user_field) {
							$custom_user_field_value = $this->validate_input($custom_user_field['value']);
							if (array_key_exists($custom_user_field_key, $billing_fields)) {
								$crm_field = $billing_fields[$custom_user_field_key];
								$data['customer']['billing'][$crm_field] = $custom_user_field_value;
							} else {
								$data['customer']['meta_data'][] = [
									'key' => strpos($custom_user_field_key, 'dbem_') === 0 ? substr($custom_user_field_key, 5) : $custom_user_field_key,
									'value' => $custom_user_field_value
								];
							}
						}
				}
				//additional meta overrides existing fields
//				foreach ($person_meta as $k => $meta) {
//					$data['customer']['meta_data'][] =
//						[
//							'key' => $k,
//							'value' => implode(' ', $meta)
//						];
//				}
			} else {

				//check if its woo checkout
				if (is_plugin_active('events-manager-woocommerce/events-manager-woocommerce.php')) {
					$checkout_fields = $checkout->get_checkout_fields();
					$posted_data = $checkout->get_posted_data();
					// why is the validate function protected?
//					$checkout->validate_posted_data($posted_data, $errors);
					foreach ($checkout_fields['billing'] as $key => $posted_datum) {
						if(isset($posted_data[$key])){
						$prefix = 'billing_';
							if(strpos($key, $prefix) === 0){
								$billing_key = substr($key, strlen($prefix));
								$data['customer']['billing'][$billing_key] = $this->validate_input($posted_data[$key]);
							}
						}
					}
				} else {
					// map EM fields and add others to meta_data
					if (!empty($EM_Booking->booking_meta['registration']) && is_array($EM_Booking->booking_meta['registration']))
						foreach ($EM_Booking->booking_meta['registration'] as $custom_user_field_key => $custom_user_field_value) {
							if (isset($custom_user_field_value)) {
								$custom_user_field_value = $this->validate_input($custom_user_field_value);
								if (array_key_exists($custom_user_field_key, $billing_fields)) {
									$crm_field = $billing_fields[$custom_user_field_key];
									$data['customer']['billing'][$crm_field] = $custom_user_field_value;
								} else {
									$data['customer']['meta_data'][] = [
										'key' => strpos($custom_user_field_key, 'dbem_') === 0 ? substr($custom_user_field_key, 5) : $custom_user_field_key,
										'value' => $custom_user_field_value
									];
								}
							}
						}
				}
			}
			return true;
		}
		return false;
		*/
    }

	private function validate_input($data) {
		$data = trim($data);
		$data = stripslashes($data);
		$data = htmlspecialchars($data);
		return $data;
	}

    /**
     * @param $data
     * @param EM_Booking $EM_Booking
     */
    protected function add_em_billing_address(&$data, $EM_Booking)
    {
        if (!empty($_REQUEST['billing_address_alternate'])) {
            $data['billing_address'] = [
//                'address' => $EM_Booking->booking_meta['registration']['dbem_address'],
//                'city' => $EM_Booking->booking_meta['registration']['dbem_city'],
//                'state' => $EM_Booking->booking_meta['registration']['dbem_state'],
//                'zip' => $EM_Booking->booking_meta['registration']['dbem_zip'],
//                'country' => $EM_Booking->booking_meta['registration']['dbem_country'],
//                'phone' => $EM_Booking->booking_meta['registration']['dbem_phone'],
                'name' => $this->validate_input($_REQUEST['wpwh']['checkout']['billing_address_name'] ?? $EM_Booking->get_person()->get_name()),
                'address' => $this->validate_input($_REQUEST['wpwh']['checkout']['billing_address_street'] ?? $EM_Booking->booking_meta['registration']['dbem_address']),
                'city' => $this->validate_input($_REQUEST['wpwh']['checkout']['billing_address_city'] ?? $EM_Booking->booking_meta['registration']['dbem_city']),
                'state' => $this->validate_input($_REQUEST['wpwh']['checkout']['billing_address_state'] ?? $EM_Booking->booking_meta['registration']['dbem_state']),
                'zip' => $this->validate_input($_REQUEST['wpwh']['checkout']['billing_address_postalcode'] ?? $EM_Booking->booking_meta['registration']['dbem_zip']),
                'country' => $this->validate_input($_REQUEST['wpwh']['checkout']['billing_address_country'] ?? $EM_Booking->booking_meta['registration']['dbem_country']),
                'phone' => $this->validate_input($_REQUEST['wpwh']['checkout']['billing_phone'] ?? $EM_Booking->booking_meta['registration']['dbem_phone']),
            ];
            $wpwh_request = $_REQUEST['wpwh']['checkout'];
            unset(
                $wpwh_request['billing_address_name'],
                $wpwh_request['billing_address_street'],
                $wpwh_request['billing_address_city'],
                $wpwh_request['billing_address_state'],
                $wpwh_request['billing_address_postalcode'],
                $wpwh_request['billing_address_country'],
                $wpwh_request['billing_phone']
            );
            $meta_data = [];
            if (!empty($wpwh_request) && is_array($wpwh_request))
                foreach ($wpwh_request as $k => $v) {
                    $meta_data[] = [
                        'key' => $this->validate_input($k),
                        'value' => $this->validate_input($v)
                    ];
                }
            if (isset($data['meta_data']))
                $data['meta_data'] = array_merge($data['meta_data'], $meta_data);
        }
    }

    /**
     * @param $data
     * @param EM_Booking $EM_Booking
     */
    protected function add_em_payment(&$data, $EM_Booking)
    {
        //TODO general payments?

        if (empty($EM_Booking->custom['payment']))
            return;

        $payment = $EM_Booking->custom['payment'];

        //sepa
        if (!empty($payment['sepa'])) {
            if (empty($data['payment']))
                $data['payment'] = [];

            $data['payment']['sepa'] = $payment['sepa'];
            unset($payment['sepa']);
        }

        $meta_data = [];
        if (is_array($payment))
            foreach ($payment as $k => $v) {
                $meta_data[] = [
                    'key' => $k,
                    'value' => $v
                ];
            }

        if (isset($data['meta_data']))
            $data['meta_data'] = array_merge($data['meta_data'], $meta_data);
    }

    /**
     * @param bool $is_valid
     * @param EM_Event $EM_Event
     * @return EM_Event|bool
     */
    public function wpwh_send_event($is_valid, $EM_Event)
    {
		//prevent loops
		global $wpwh;
		if (isset($wpwh['do_not_send']) && $wpwh['do_not_send']) {
			return false;
		}

        global $post;

        // only proceed if sync_events isset
        if (!$this->wpwh_options['sync_events'])
            return $EM_Event;

        //wpml: don't sync translated event
		if(isset($EM_Event->event_translation) && $EM_Event->event_translation)
			return $EM_Event;

		//don't sync recurring events (for now)
		if($EM_Event->is_recurring() || $EM_Event->is_recurrence())
			return $EM_Event;

        if ($is_valid) {
            /**
             * Date/Time UTC
             */
			$EM_DateTime_start = $EM_Event->start();
			$EM_DateTime_end = $EM_Event->end();
			$EM_DateTime_rsvp_end = $EM_Event->rsvp_end();
			// set UTC manually, because $EM_Event->start(true) had sometimes Berlin and sometimes UTC
			$EM_DateTime_UTC_start = $EM_DateTime_start->setTimezone('UTC');
			$EM_DateTime_UTC_end = $EM_DateTime_end->setTimezone('UTC');
			$EM_DateTime_UTC_rsvp_end = $EM_DateTime_rsvp_end->setTimezone('UTC');

            // event
            $data = [
                'id' => $EM_Event->post_id,
                'status' => $EM_Event->event_status,
                'name' => $EM_Event->event_name,
                'slug' => $EM_Event->event_slug,
                'all_day' => $EM_Event->event_all_day,
                'start_date_time_utc' => $EM_DateTime_UTC_start->format('Y-m-d H:i:s'),
                'end_date_time_utc' => $EM_DateTime_UTC_end->format('Y-m-d H:i:s'),
                'rsvp_date_time_utc' => $EM_DateTime_UTC_rsvp_end->format('Y-m-d H:i:s'),
                'spaces' => $EM_Event->event_spaces ? $EM_Event->event_spaces : '',
                'description' => !empty($EM_Event->post_content) ? $EM_Event->post_content : $post->post_content,
                'url' => get_permalink($EM_Event->post_id),
                'rsvp' => $EM_Event->event_rsvp,
            ];

            /**
             * categories
             * @var EM_Categories $EM_Categories
             * @var EM_Category $EM_Category
             */
            $EM_Categories = $EM_Event->get_categories();
            foreach ($EM_Categories as $EM_Category) {
                $data['categories'][] = [
                    "id" => $EM_Category->id,
                    "term_id" => $EM_Category->term_id,
                    "name" => $EM_Category->name,
                    "slug" => $EM_Category->slug,
                    "term_group" => $EM_Category->term_group,
                    "term_taxonomy_id" => $EM_Category->term_taxonomy_id,
                    "description" => $EM_Category->description,
                    "parent" => $EM_Category->parent
                ];
            }

            /**
             * tags
             */
            $tags = get_the_terms($EM_Event->post_id, 'event-tags');
            if (!empty($tags))
                foreach ($tags as $tag) {
                    $data['tags'][] = [
                        "term_id" => $tag->term_id,
                        "name" => $tag->name,
                        "slug" => $tag->slug,
                        "term_group" => $tag->term_group,
                        "term_taxonomy_id" => $tag->term_taxonomy_id,
                        "description" => $tag->description,
                        "parent" => $tag->parent
                    ];
                }

            /**
             * location
             */
            $EM_Location = $EM_Event->get_location();
            if (!empty($EM_Location))
                $data['location'] = [
                    'id' => $EM_Location->post_id,
                    'status' => $EM_Location->location_status,
                    'name' => $EM_Location->location_name,
                    'slug' => $EM_Location->location_slug,
                    'address' => $EM_Location->location_address,
                    'country' => $EM_Location->location_country,
                    'latitude' => $EM_Location->location_latitude,
                    'longitude' => $EM_Location->location_longitude,
                    'postcode' => $EM_Location->location_postcode,
                    'region' => $EM_Location->location_region,
                    'state' => $EM_Location->location_state,
                    'town' => $EM_Location->location_town
                ];

            /**
             * tickets
             * @var EM_Tickets $EM_Tickets
             * @var EM_Ticket $EM_Ticket
             */
            $EM_Tickets = $EM_Event->get_tickets();
            if (!empty($EM_Tickets))
                foreach ($EM_Tickets->tickets as $EM_Ticket) {
                    $data['tickets'][] = [
                        'id' => $EM_Ticket->ticket_id,
                        'name' => $EM_Ticket->ticket_name,
                        'description' => $EM_Ticket->ticket_description,
                        'price' => $EM_Ticket->ticket_price,
                        'start' => $EM_Ticket->ticket_start,
                        'end' => $EM_Ticket->ticket_end,
                        'spaces' => $EM_Ticket->ticket_spaces,
                        'spaces_available' => $EM_Ticket->ticket_spaces - $EM_Ticket->get_booked_spaces(),
                        'spaces_irrelevant' => $EM_Ticket->ticket_meta['event_spaces_irrelevant'],
						'roles' => $EM_Ticket->ticket_members ? $EM_Ticket->ticket_members_roles : []
                    ];
                }

            //if legacy v3
            if ($this->wpwh_options['use_legacy_v3'])
                $data = ['event' => $data];

            // custom event data
            $data = apply_filters('wpwh_event_data', $data, $EM_Event);

            // fire webhook
            try {
                $json = json_encode($data);
                //$result = v4WebHook::fire_webhook($json, 'created_updated');
                v4WebHook::fire_webhook($json, 'event', 'created_updated');
            } catch (Exception $e) {
                //$result = false;
                $this->add_admin_notice('1CRM event synchronisation failed. ' . $e->getMessage(), true);
            }
        }
        return $EM_Event;
    }

    /**
     * send custom data
     *
     * @param array $data
     * @return bool|string
     */
    public function wpwh_send_cstm_data($data, $resource = "custom")
    {
		//prevent loops
		global $wpwh;
		if (isset($wpwh['do_not_send']) && $wpwh['do_not_send']) {
			return false;
		}

        if (is_array($data) && !empty($data)) {
            // fire webhook
            try {
                $json = json_encode($data);
                $result = v4WebHook::fire_webhook($json, $resource);
            } catch (Exception $e) {
                $result = $e->getMessage();
            }
        } else
            $result = 'invalid data';

        return $result;
    }

    /**
     * fire webhook
     *
     * @param $json
     * @param $resource
     * @param $event
     * @return bool
     * @throws Exception
     */
//    public function fire_webhook($json, $resource, $event = 'created')
//    {
//        $url = rtrim($this->wpwh_options['1crm_url'], "/") . '/json.php?module=v4WebHook&action=woowh_request';
//        if ($_SERVER['SERVER_NAME'] == 'wp.localdev.visual4.de') // for testing
//            $url .= '&XDEBUG_SESSION_START=PHPSTORM';
//
//        //add cookie consents
//        $json_obj = json_decode($json);
//        $v4core_cookie_setting_necessary = (isset($_COOKIE['v4core_cookie_setting_necessary']) && $_COOKIE['v4core_cookie_setting_necessary'] == 1);
//        if(!is_admin() && $v4core_cookie_setting_necessary){
//            if(isset($_COOKIE['v4core_cookie_setting_user_id']))
//                $json_obj->cookie_settings['user_id'] = $_COOKIE['v4core_cookie_setting_user_id'];
//            $json_obj->cookie_settings['is_necessary_set'] = $v4core_cookie_setting_necessary;
//            $json_obj->cookie_settings['is_preferences_set'] = (isset($_COOKIE['v4core_cookie_setting_preferences']) && $_COOKIE['v4core_cookie_setting_preferences'] == 1);
//            $json_obj->cookie_settings['is_statistics_set'] = (isset($_COOKIE['v4core_cookie_setting_statistics']) && $_COOKIE['v4core_cookie_setting_statistics'] == 1);
//            $json_obj->cookie_settings['is_marketing_set'] = (isset($_COOKIE['v4core_cookie_setting_marketing']) && $_COOKIE['v4core_cookie_setting_marketing'] == 1);
//        }
//        $json = json_encode($json_obj);
//
//        $secret = $this->wpwh_options['secret'];
//        $hash_raw = hash_hmac('sha256', $json, $secret, true);
//        $hash = base64_encode($hash_raw);
//
//        //build header
//        $topic = $resource . '.' . $event;
//        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' || $_SERVER['SERVER_PORT'] == 443) ? "https://" : "http://";
//        $header = [
//            'x-wc-webhook-signature: ' . $hash,
//            'x-wc-webhook-topic: ' . $topic,
//            'x-wc-webhook-resource: ' . $resource,
//            'x-wc-webhook-event: ' . $event,
//            'x-wc-webhook-source: ' . $protocol . $_SERVER['HTTP_HOST']
//        ];
//
//        // setup curl
//        $curl = curl_init();
//        curl_setopt($curl, CURLOPT_URL, $url);
//        curl_setopt($curl, CURLOPT_HTTPHEADER, $header);
//        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1); //disable if asynchronous
//        curl_setopt($curl, CURLOPT_POST, 1);
//        curl_setopt($curl, CURLOPT_POSTFIELDS, $json);
//        curl_setopt($curl, CURLOPT_TIMEOUT, 30);
////        curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, (get_option('wpoa_http_util_verify_ssl') == 1 ? 1 : 0));
////        curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, (get_option('wpoa_http_util_verify_ssl') == 1 ? 2 : 0));
//        $response = curl_exec($curl);
//        $error = curl_error($curl);
//        $info = curl_getinfo($curl);
//        curl_close($curl);
//
//        if (is_string($response))
//            $response = trim($response, "\n");
//
//        $this->wpwh_log($json, $response, $info['http_code'], $error);
//        if ($response != false && empty($error) && $info['http_code'] == '200') {
////            $this->wpwh_log(" - Response: $response \n - Request: $json", false);
//            return true;
//        } else {
////            $this->wpwh_log(" - Status: {$info['http_code']} \n - Response: $response \n - Request: $json \n - cURL_error: $error");
//            $msg = (!empty($response) ? $response : $error);
//            throw new \Exception(!empty($msg) ? $msg : ('(' . $info['http_code'] . ')'), $info['http_code']);
//        }
//    }
//
//    /**
//     * @param $request
//     * @param $response
//     * @param int $http_code
//     * @param string $error
//     * @var wpdb $wpdb
//     */
//    private function wpwh_log($request, $response, $http_code = 200, $error = '')
//    {
//        //backup old logs
//        $files = [
//            __DIR__ . '/logs/v4-wp-webhook-error.log',
//            __DIR__ . '/logs/v4-wp-webhook.log'
//        ];
//        foreach ($files as $file)
//            if (file_exists($file))
//                rename($file, $file . '.' . uniqid());
//
//        global $wpdb;
//        $date = date('Y-m-d H:i:s');
//
//        // create table if not exists
//        $query = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}wpwh_log (###ID###, datetime datetime, http_code integer, request text, response text, error varchar(255)###AFTER###)";
//        $mysql_query = str_replace(['###ID###', '###AFTER###'], ['id int NOT NULL AUTO_INCREMENT', ', PRIMARY KEY (id)'], $query);
//        $sql_query = str_replace(['###ID###', '###AFTER###'], ['id int IDENTITY(1,1) PRIMARY KEY', ''], $query);
//        $mysql_result = $wpdb->query($wpdb->prepare($mysql_query, []));
//        if ($mysql_result == false)
//            $wpdb->query($wpdb->prepare($sql_query, []));
//
//        // insert
//        $wp_debug_msg = 'Webhook body is not logged unless WP_DEBUG mode is turned on. This is to avoid the storing of personal data in the logs.';
//        $data = [
//            'datetime' => $date,
//            'http_code' => (int)$http_code,
//            'request' => (defined('WP_DEBUG') && true === WP_DEBUG) ? $request : $wp_debug_msg,
//            'response' => $response
//        ];
//        $format = [
//            '%s',
//            '%d',
//            '%s',
//            '%s'
//        ];
//        if (!empty($error)) {
//            $data['error'] = $error;
//            $format[] = '%s';
//        }
//        $wpdb->insert("{$wpdb->prefix}wpwh_log", $data, $format);
//    }

////
/// Custom Register Fields
///

	public function validate_recaptcha()
	{
		$is_valid = false;

		$g_recaptcha_response = $_POST['g-recaptcha-response'];

		$url = 'https://www.google.com/recaptcha/api/siteverify';
		$data = array(
            'secret' => $this->wpwh_options['recaptcha_secret_key'],
            'response' => $g_recaptcha_response
        );

        // use key 'http' even if you send the request to https://...
		$options = array(
			'http' => array(
				'header'  => "Content-type: application/x-www-form-urlencoded\r\n",
				'method'  => 'POST',
				'content' => http_build_query($data)
			)
		);
		$context  = stream_context_create($options);
		$result = file_get_contents($url, false, $context);
		if ($result !== FALSE) {
		    $json = json_decode($result);
		    if($json->success === true)
				$is_valid = true;
        }

		return $is_valid;
	}

	public function template_recaptcha()
	{
		$recaptcha_site_key = $this->wpwh_options['recaptcha_site_key'] ?? false;
		$recaptcha_secret_key = $this->wpwh_options['recaptcha_secret_key'] ?? false;

		if (!empty($recaptcha_site_key) && !empty($recaptcha_secret_key)) {
            $recaptcha = '<div class="g-recaptcha" data-sitekey="'.$recaptcha_site_key.'"></div>';
            echo apply_filters('v4-modify-template-recaptcha', $recaptcha);
		}
	}

	public function wc_validate_recaptcha($data, $errors)
	{
		$is_valid = $this->validate_recaptcha();
		if (!$is_valid) {
			$error_msg = sprintf(esc_html__('Please enter a valid %1$s', 'v4-wp-webhook'), 'reCaptcha');
			$errors->add('recaptcha', $error_msg);
		}
	}

	public function display_tools()
	{
		include __DIR__ . '/partials/v4-wp-webhook-admin-display--tools.php';
	}

	public function cleanup_serialized_meta()
	{
		ob_clean();

		$result = v4Customer::cleanup_serialized_meta_data();

		if ($result === false) {
			http_response_code(400);
			echo 'sorry, you cannot do this.';
		} elseif ($result === 0) {
			http_response_code(200);
			echo 'done!';
		} else {
			http_response_code(202); // 100 doesn't work... Browser gets 200 instead
			echo $result;
		}

		die();
	}

}
