<?php

require_once __DIR__ . '/aWPWH_REST_EMP.php';

class WPWH_REST_EMP_Booking extends aWPWH_REST_EMP
{

    public function __construct()
    {
        parent::__construct('bookings');
    }

    /**
     * Register the routes for the objects of the controller.
     */
    public function register_routes()
    {
        register_rest_route($this->namespace, '/' . $this->base, array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_items'),
                'permission_callback' => array($this, 'get_items_permissions_check'),
                'args' => array(),
            ),
            array(
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => array($this, 'create_update_item'),
                'permission_callback' => array($this, 'create_item_permissions_check'),
            ),
        ));
        register_rest_route($this->namespace, '/' . $this->base . '/(?P<id>[\d]+)', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_item'),
                'permission_callback' => array($this, 'get_item_permissions_check'),
                'args' => array(
                    'context' => array(
                        'default' => 'view',
                    ),
                ),
            ),
            array(
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => array($this, 'create_update_item'),
                'permission_callback' => array($this, 'update_item_permissions_check'),
            ),
            array(
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => array($this, 'delete_item'),
                'permission_callback' => array($this, 'delete_item_permissions_check'),
                'args' => array(
                    'force' => array(
                        'default' => false,
                    ),
                ),
            ),
        ));
//        register_rest_route($this->namespace, '/' . $this->base . '/schema', array(
//            'methods' => WP_REST_Server::READABLE,
//            'callback' => array($this, 'get_public_item_schema'),
//        ));
    }

    /**
     * Get a collection of items
     *
     * @param WP_REST_Request $request Full data about the request.
     * @return WP_Error|WP_REST_Response
     */
    public function get_items($request)
    {
        $data = array(
            'bookings' => 'allowed'
        );

        return new WP_REST_Response($data, 200);
    }

    /**
     * Get one item from the collection
     *
     * @param WP_REST_Request $request Full data about the request.
     * @return WP_Error|WP_REST_Response
     */
    public function get_item($request)
    {
        //get parameters from request
        $params = $request->get_params();

        $EM_Booking = em_get_booking($params['id']);

        //return a response or error based on some conditional
        if ($EM_Booking->validate()) {
            $data = $this->prepare_item_for_response($EM_Booking);
            return new WP_REST_Response($data, 200);
        } else {
            return new WP_Error('not-found', __('Booking'));
        }
    }

    /**
     * Create or Update one item from the collection
     *
     * @param WP_REST_Request $request Full data about the request.
     * @return WP_Error|WP_REST_Response
     */
    public function create_update_item($request)
    {
        global $wpwh;

        $data = $this->prepare_item_for_database($request);
        $EM_Booking = em_get_booking($request['id']);
        $type = empty($EM_Booking->booking_id) ? 'create' : 'update';

        try {
            $this->add_booking_data($EM_Booking, $data);
        } catch (Exception $e) {
            if($type == 'create')
                $EM_Booking->delete();
            if(!empty($this->error))
                return new WP_Error($this->error['code'], $this->error['message'], array('status' => $this->error['status']));
            else
                return new WP_Error('cant-' . $type, $e->getMessage(), array('status' => $e->getCode()));
        }

		// validate without tickets (override_availability true)
        if (!$EM_Booking->validate(true)) {
			$errors = $EM_Booking->get_errors();
			//might be not valid, but only because of missing tickets. in that case we can save, because tickets will be synced separately
			if(!empty($errors)){
				return new WP_Error('cant-' . $type, print_r($errors, true), array('status' => 500));
			}
		}

        $sendMail = true;
        $EM_Event = $EM_Booking->get_event();
        if ($EM_Event->end()->getTimestamp() <= time()) {
            $sendMail = false;
        }


        //create booking, if not exists
		if(!$EM_Booking->booking_id){
			$wpwh['do_not_send'] = true;
			$EM_Booking->save($sendMail);
			unset($wpwh['do_not_send']);
		}

		//tickets_bookings // allow without tickets, because tickets will be added later
		if (!empty($data->tickets_bookings)) {
			$EM_Tickets_Bookings = $EM_Booking->get_tickets_bookings();

			//delete existing tickets (override not possible without id (thx EM v6.1))
			if(!empty($EM_Tickets_Bookings->tickets_bookings)) {
				$EM_Tickets_Bookings->tickets_bookings_deleted = $EM_Tickets_Bookings->tickets_bookings;
				$wpwh['do_not_send'] = true;
				$EM_Tickets_Bookings->save();
				unset($wpwh['do_not_send']);
				$EM_Tickets_Bookings->tickets_bookings_deleted = [];
				$EM_Tickets_Bookings->tickets_bookings = [];
			}

			//reset tickets
			$new_spaces = 0;
			foreach ($data->tickets_bookings as $crm_tickets_booking) {
				// creation doesn't work on new Bookings, because of missing booking_id.
				// We can do this afterwards via api ticket_booking.
				// but this part is necessary for existing bookings or after sync errors.
				$EM_Ticket_Booking_data = array(
					'booking_id' => $EM_Booking->booking_id,
					'ticket_id' => $crm_tickets_booking->ticket->id,
					'ticket_booking_spaces' => (int) $crm_tickets_booking->spaces,
					'ticket_booking_price' => $crm_tickets_booking->price
				);
				$EM_Ticket_Booking = new EM_Ticket_Booking($EM_Ticket_Booking_data);
				$EM_Tickets_Bookings->add($EM_Ticket_Booking);
				$new_spaces += (int) $crm_tickets_booking->spaces;
			}

			//update spaces
			$EM_Tickets_Bookings->get_spaces(true);
//			$EM_Tickets_Bookings->spaces = $new_spaces;

			$EM_Booking->tickets_bookings = $EM_Tickets_Bookings;
		}

		//update spaces and prices
		$wpwh['do_not_send'] = true;
		$EM_Booking->get_spaces(true);
		$EM_Booking->calculate_price();
		$EM_Booking->save($sendMail);
		unset($wpwh['do_not_send']);

		if(!empty($EM_Booking->errors))
			return new WP_REST_Response(implode(', ', $EM_Booking->errors), 500);

        $data = $this->prepare_item_for_response($EM_Booking);
        return new WP_REST_Response($data, 200);
    }

    /**
     * @param EM_Booking $EM_Booking
     * @param $data
     * @throws Exception
     */
    private function add_booking_data($EM_Booking, $data)
    {
        global $wpwh;

        //status
        if ($data->status != $EM_Booking->booking_status) {
            $wpwh['do_not_send'] = true;
            $EM_Booking->set_status($data->status, false);
            unset($wpwh['do_not_send']);
        }

        //general
        $EM_Booking->booking_price = $data->price;
        $EM_Booking->booking_spaces = $data->spaces;
        $EM_Booking->manage_override = true; //admin is booking

        //meta json Obj to array
        $meta_array = array();
        foreach ($data->meta as $k => $v)
            $meta_array[$k] = $v;
        $EM_Booking->booking_meta = array_merge($EM_Booking->booking_meta, $meta_array);

        //event
        $EM_Booking->event = em_get_event($data->event->id, 'post_id');
        $EM_Booking->event_id = $EM_Booking->event->event_id;

        //person
        $EM_Booking->person = new EM_Person($data->user->id);
        $EM_Booking->person_id = $EM_Booking->person->ID;
		$EM_Booking->person->first_name = $data->user->first_name;
		$EM_Booking->person->last_name = $data->user->last_name;
		$EM_Booking->person->user_email = $data->user->email;

//		$EM_Booking->booking_meta['registration'] = $data->user;
		$EM_Booking->booking_meta['registration']['user_email'] = $data->user->email;
        //TEMPORÄRER BUGFIX: Benutzerdefinierte Pflichtfelder im Buchungsformular an die Validation vom EM zu übergeben
        //em_booking_validate wp-content/plugins/events-manager-pro/add-ons/bookings-form/bookings-form.php:229
		$EM_Booking->booking_meta['booking'] = $meta_array;
//		$_REQUEST['user_email'] = $_POST['user_email'] = $data->user->email;


        //TODO add address? not necessary for now
    }

    /**
     * Delete one item from the collection
     *
     * @param WP_REST_Request $request Full data about the request.
     * @return WP_Error|WP_REST_Response
     */
    public function delete_item($request)
    {
		global $wpwh;
        $wpwh['do_not_send'] = true;

        $EM_Booking = em_get_booking($request['id']);
        if (is_a($EM_Booking, 'EM_Booking'))
            if ($EM_Booking->delete())
                return new WP_REST_Response(true, 200);

        return new WP_Error('cant-delete', __('message', 'text-domain'), array('status' => 500));
    }

    /**
     * Prepare the item for create or update operation
     *
     * @param WP_REST_Request $request Request object
     * @return WP_Error|object
     */
    protected function prepare_item_for_database($request)
    {
        $body = $request->get_body();
        $data = json_decode($body);
        return $data;
    }

    /**
     * Prepare the item for the REST response
     *
     * @param EM_Booking $EM_Booking
     * @return mixed
     */
    public function prepare_item_for_response($EM_Booking)
    {
        $data = $this->v4_wp_webhook_admin->booking_webhook_data($EM_Booking);
        
        $data['notices'] = $this->notices;
        
        return $data;
    }
}