<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       http://www.visual4.de/
 * @since      1.0.0
 *
 * @package    V4_Wp_Webhook
 * @subpackage V4_Wp_Webhook/includes
 */

require_once __DIR__ . '/v4Webhook.php';

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    V4_Wp_Webhook
 * @subpackage V4_Wp_Webhook/includes
 * @author     visual4 <info@visual4.de>
 */
class V4_Wp_Webhook
{

    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      V4_Wp_Webhook_Loader $loader Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string $plugin_name The string used to uniquely identify this plugin.
     */
    protected $plugin_name;

    /**
     * The current version of the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string $version The current version of the plugin.
     */
    protected $version;

    private $settings;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function __construct()
    {

        $this->plugin_name = 'v4-wp-webhook';
        $this->version = '1.0.0';

        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();

        $this->settings = new V4_Wp_Webhook_Settings();

//        global $wpwh;
//        $wpwh = new V4_Wp_Webhook_Helper();
    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Include the following files that make up the plugin:
     *
     * - V4_Wp_Webhook_Loader. Orchestrates the hooks of the plugin.
     * - V4_Wp_Webhook_i18n. Defines internationalization functionality.
     * - V4_Wp_Webhook_Admin. Defines all hooks for the admin area.
     * - V4_Wp_Webhook_Public. Defines all hooks for the public side of the site.
     *
     * Create an instance of the loader which will be used to register the hooks
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function load_dependencies()
    {

        /**
         * The class responsible for orchestrating the actions and filters of the
         * core plugin.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-v4-wp-webhook-loader.php';

        /**
         * The class responsible for defining internationalization functionality
         * of the plugin.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-v4-wp-webhook-i18n.php';

        /**
         * Settings
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-v4-wp-webhook-settings.php';

        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-v4-wp-webhook-helper.php';

        /**
         * The class responsible for defining all actions that occur in the admin area.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-v4-wp-webhook-admin.php';

        /**
         * The class responsible for defining all actions that occur in the public-facing
         * side of the site.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'public/class-v4-wp-webhook-public.php';

        $this->loader = new V4_Wp_Webhook_Loader();

    }

    /**
     * Define the locale for this plugin for internationalization.
     *
     * Uses the V4_Wp_Webhook_i18n class in order to set the domain and to register the hook
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function set_locale()
    {

        $plugin_i18n = new V4_Wp_Webhook_i18n();

        $this->loader->add_action('plugins_loaded', $plugin_i18n, 'load_plugin_textdomain');

    }

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_admin_hooks()
    {

        $plugin_admin = new V4_Wp_Webhook_Admin($this->get_plugin_name(), $this->get_version());

        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_styles');
        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts');

        /**
         * webhooks
         */
//        $this->loader->add_action('wp', $plugin_admin, 'wpwh_test', 999); //priority last
//        $this->loader->add_action('user_register', $plugin_admin, 'wpwh_send_user', 999); //priority last

        /** send booking */
        if (get_option('dbem_multiple_bookings')) {
            $this->loader->add_action('em_multiple_booking_save', $plugin_admin, 'wpwh_send_booking', 999, 2); //priority last
        } else {
            $this->loader->add_action('em_booking_save', $plugin_admin, 'wpwh_send_booking', 999, 2); //priority last
        }
        $this->loader->add_action('em_booking_set_status', $plugin_admin, 'wpwh_send_booking', 999, 2); //priority last
        $this->loader->add_action('em_booking_delete', $plugin_admin, 'wpwh_send_booking', 999, 2); //priority last

        /** send custom data */
        $this->loader->add_filter('wpwh_send_cstm_data', $plugin_admin, 'wpwh_send_cstm_data', 999, 2); //priority last

        /** send event */
        $this->loader->add_filter('em_event_save', $plugin_admin, 'wpwh_send_event', 999, 2); //priority last

        /** 1crm account */
        $this->loader->add_action('admin_post_wpwh_my_account', $plugin_admin, 'wpwh_my_account_admin_post');
        /** 1crm register */
        $this->loader->add_action('admin_post_nopriv_wpwh_register', $plugin_admin, 'wpwh_register_admin_post_nopriv');

        /* Add irrelevant tickets */
        $this->loader->add_action( 'em_ticket_save_pre', $plugin_admin, 'add_custom_data_to_ticket' );
        $this->loader->add_action( 'em_booking_get_available_spaces', $plugin_admin, 'get_booking_relevant_spaces', 90, 2 );
        $this->loader->add_action( 'em_ticket_get_available_spaces', $plugin_admin, 'get_ticket_relevant_spaces', 90, 2 );
//        $this->loader->add_action( 'em_booking_get_spaces', $plugin_admin, 'get_booking_spaces', 90, 2 );
        $this->loader->add_action( 'em_bookings_get_available_tickets', $plugin_admin, 'get_available_tickets', 90, 2 );


        /*
         **********************************
         * allow bookings without tickets *
         **********************************
         */
        $this->loader->add_filter('em_booking_validate', $plugin_admin, 'em_booking_validate_without_tickets', 100, 2);

        /*
         **********************************
         * allow overide templates *
         **********************************
         */
        $this->loader->add_filter('em_locate_template', $plugin_admin, 'locate_template', 100, 2);

    }

    /**
     * Register all of the hooks related to the public-facing functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_public_hooks()
    {

        $plugin_public = new V4_Wp_Webhook_Public($this->get_plugin_name(), $this->get_version());

        $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_styles');
        $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_scripts');

        /** init */
        $this->loader->add_action('init', $plugin_public, 'init', 5);

        /**
         * shortcodes
         */
        $this->loader->add_action('plugins_loaded', $plugin_public, 'v4_shortcodes');

        /** EM checkout */
        // TODO make optional
        if(class_exists('WPOA')){
            if(get_option('dbem_multiple_bookings'))
                $this->loader->add_action('em_checkout_form_footer', $plugin_public, 'v4_em_checkout_custom_fields', 2);
            else
                $this->loader->add_action('em_booking_form_custom', $plugin_public, 'v4_em_checkout_custom_fields', 2);
        }

        /** add Billing Details title */
//        $this->loader->add_action('em_booking_form_before_user_details', $plugin_public, 'v4_add_title_billing_details', 23, 1);

    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    1.0.0
     */
    public function run()
    {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @since     1.0.0
     * @return    string    The name of the plugin.
     */
    public function get_plugin_name()
    {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since     1.0.0
     * @return    V4_Wp_Webhook_Loader    Orchestrates the hooks of the plugin.
     */
    public function get_loader()
    {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @since     1.0.0
     * @return    string    The version number of the plugin.
     */
    public function get_version()
    {
        return $this->version;
    }

}
