<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       http://www.visual4.de/
 * @since      1.0.0
 *
 * @package    V4_Wp_Webhook
 * @subpackage V4_Wp_Webhook/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    V4_Wp_Webhook
 * @subpackage V4_Wp_Webhook/public
 * @author     visual4 <info@visual4.de>
 */
class V4_Wp_Webhook_Public
{

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string $plugin_name The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string $version The current version of this plugin.
     */
    private $version;

	private $wpwh_options = [];

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param      string $plugin_name The name of the plugin.
     * @param      string $version The version of this plugin.
     */
    public function __construct($plugin_name, $version)
    {

        $this->plugin_name = $plugin_name;
        $this->version = $version;

    }

    /**
     * Register the stylesheets for the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function enqueue_styles()
    {

        /**
         * This function is provided for demonstration purposes only.
         *
         * An instance of this class should be passed to the run() function
         * defined in V4_Wp_Webhook_Loader as all of the hooks are defined
         * in that particular class.
         *
         * The V4_Wp_Webhook_Loader will then create the relationship
         * between the defined hooks and the functions defined in this
         * class.
         */

        wp_enqueue_style($this->plugin_name . '-public', plugin_dir_url(__FILE__) . 'css/v4-wp-webhook-public.css', array(), $this->version, 'all');
        wp_enqueue_style($this->plugin_name . '-public-loading', plugin_dir_url(__FILE__) . 'css/v4-wp-webhook-public-loading.css', array(), $this->version, 'all');

    }

    /**
     * Register the JavaScript for the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts()
    {

        /**
         * This function is provided for demonstration purposes only.
         *
         * An instance of this class should be passed to the run() function
         * defined in V4_Wp_Webhook_Loader as all of the hooks are defined
         * in that particular class.
         *
         * The V4_Wp_Webhook_Loader will then create the relationship
         * between the defined hooks and the functions defined in this
         * class.
         */

        wp_enqueue_script($this->plugin_name . '-public', plugin_dir_url(__FILE__) . 'js/v4-wp-webhook-public.js', array('jquery'), $this->version, false);

        $recaptcha_site_key = $this->wpwh_options['recaptcha_site_key'] ?? false;
		$recaptcha_secret_key = $this->wpwh_options['recaptcha_secret_key'] ?? false;
		if (!empty($recaptcha_site_key) && !empty($recaptcha_secret_key)) {
			wp_enqueue_script('google_recaptcha', 'https://www.google.com/recaptcha/api.js');
		}

    }

    public function init()
    {
		$this->wpwh_options = get_option('v4_wpwh');
        $actions_to_validate = [
            'booking_add'
        ];
        if (!empty($_REQUEST['action']) && in_array($_REQUEST['action'], $actions_to_validate)) {
            add_filter('em_booking_validate', [$this, 'validate_checkout'], 5, 2);
        }

		// Use v4 WP WebHook Register Fields
		if(isset($this->wpwh_options['use_wpwh_register_fields']) && $this->wpwh_options['use_wpwh_register_fields']){
			remove_action('em_booking_form_custom', array('EM_Booking_Form','booking_form'));
			add_action('em_booking_form_custom', [$this, 'v4_em_booking_form_custom'], 2);
		}
    }

    /**
     * @param $result
     * @param EM_Booking $EM_Booking
     * @return bool
     */
    public function validate_checkout($result, $EM_Booking)
    {
        $missing_fields = [];
        if (!empty($_REQUEST['billing_address_alternate']))
            foreach ($_REQUEST['wpwh']['checkout-required'] as $k => $v)
                if (empty($_REQUEST['wpwh']['checkout'][$k]))
                    $missing_fields[] = $v;

        if (!empty($missing_fields)){
            $result = false;
            $EM_Booking->add_error(__('Missing fields: ', 'events-manager') . ' ' . implode(", ", $missing_fields));
        }
        //only check recaptcha on post to prevent errors
		$recaptcha_site_key = $this->wpwh_options['recaptcha_site_key'] ?? false;
		$recaptcha_secret_key = $this->wpwh_options['recaptcha_secret_key'] ?? false;
		if (!is_user_logged_in() && !empty($recaptcha_site_key) && !empty($recaptcha_secret_key) && empty(apply_filters('v4_remove_recaptcha_from_em_booking', null)) && !empty($_POST)) {
			/** @var V4_Wp_Webhook_Admin $V4_Wp_Webhook_Admin */
			global $V4_Wp_Webhook_Admin;
			$is_valid = $V4_Wp_Webhook_Admin->validate_recaptcha();
			if(!$is_valid){
				$result = false;
				$EM_Booking->add_error(sprintf(__('Please fill in the field: %s', 'em-pro'), 'reCaptcha'));
			}
		}

		if($this->wpwh_options['use_wpwh_register_fields'] && !empty($_POST)){
			global $V4_Wp_Webhook_Admin;
			$user = [];
			$register_errors = [];
			$V4_Wp_Webhook_Admin->validate_register_fields($user, $register_errors);

			$is_logged_in = is_user_logged_in();
			$show_privacy_if_logged_in = isset($this->wpwh_options['show_privacy_if_logged_in']) && $this->wpwh_options['show_privacy_if_logged_in'];
			$validate_privacy = !$is_logged_in || $show_privacy_if_logged_in;
			if ($validate_privacy)
				$V4_Wp_Webhook_Admin->validate_consent($register_errors);

			$EM_Form = EM_Booking_Form::get_form($EM_Booking->event_id, $EM_Booking);
			$EM_Booking->get_person()->user_email = $user['email'];
			$EM_Booking->booking_meta['registration'] = $user;
			$EM_Booking->booking_meta['registration']['user_email'] = $user['email'];
			$EM_Form->field_values['user_email'] = $user['email'];
			$_REQUEST['user_email'] = $_POST['user_email'] = $user['email'];

			if(!empty($register_errors)){
				//error
				foreach ($register_errors as $k => $v) {
					$EM_Booking->add_error($v);
				}
				return false;
			}
		}

        return $result;
    }

//    public function v4_add_title_billing_details($EM_Event){
//        echo '<h3>' . _e('Billing Details', 'woocommerce') . '</h3>';
//    }

    /**
     * shortcodes
     */
    public function v4_shortcodes()
    {
        add_shortcode('wpwh_login', array( $this, 'wpwh_login_func'));
		add_shortcode('wpwh_register', array( $this, 'wpwh_register_func'));
		add_shortcode('wpwh_my_account', array( $this, 'wpwh_my_account_func'));
        add_shortcode('wpwh_my_account_register', array( $this, 'wpwh_my_account_register_func'));
    }

    public function wpwh_login_func($args)
    {
        ob_start();
        include 'partials/v4-wp-webhook-public-login.php';
        return ob_get_clean();
    }

    public function wpwh_register_func($args)
    {
        ob_start();
        include 'partials/v4-wp-webhook-public-register.php';
        return ob_get_clean();
    }

    public function wpwh_my_account_func($args)
    {
        ob_start();
        include 'partials/v4-wp-webhook-public-my-account.php';
        return ob_get_clean();
    }

	/**
	 * @param $args
	 * @return false|string
	 * @deprecated use [wpwh_my_account register="1"] instead
	 */
    public function wpwh_my_account_register_func($args)
    {
		return do_shortcode('[wpwh_my_account register="1"]');
    }

    /**
     * @param EM_Event|EM_Multiple_Booking $EM_Obj
     */
    public function v4_em_checkout_custom_fields($EM_Obj){
        include 'partials/v4-wp-webhook-public-checkout.php';
    }

    /**
     * @param EM_Event|EM_Multiple_Booking $EM_Obj
     */
    public function v4_em_booking_form_custom($EM_Obj){
		global $V4_Wp_Webhook_Admin;
		$account_fields = $V4_Wp_Webhook_Admin->get_my_account_fields();

		$user = [];
		if(is_user_logged_in()){
			foreach ($account_fields as $k => $account_field) {
				switch ($k) {
					case 'email':
						$current_user = wp_get_current_user();
						$value = $current_user->data->user_email;
						break;
					default:
						$value = $k ? maybe_unserialize(get_user_meta(get_current_user_id(), $k, true)) : '';
						break;
				}
				$user[$k] = $value;
			}
		}

        include 'partials/v4-wp-webhook-public-register-fields.php';


		$is_logged_in = is_user_logged_in();
		$show_privacy_if_logged_in = isset($this->wpwh_options['show_privacy_if_logged_in']) && $this->wpwh_options['show_privacy_if_logged_in'];
		$validate_privacy = !$is_logged_in || $show_privacy_if_logged_in;
		if ($validate_privacy)
			include 'partials/v4-wp-webhook-public-consent-fields.php';

    }

}
